



################################################################
# CLASS: e_fsm
#
################################################################


=head1 NAME

e_fsm - description of the module goes here ...

=head1 SYNOPSIS

The e_fsm class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_fsm;
@ISA = qw(e_module);
use europa_all;
use europa_utils;
use strict;

################################################################
#
# fsm 
#
# Finite State Machine generator
#
# Usage:
#
# Create a new, empty state machine:
# my $fsm = e_fsm->new();
#
# Name the state machine (not strictly required, if only one
#  machine exists, but its name is used in the state machine's
#  reg declarations - therefore it must be a legal identifier
#  in the output language):
# $fsm->name("my_little_state_machine");
#
# Set the state machine output default values:
# my %defaults = (
#   output1 => '0',
#   output2 => "1'bX",
#   output3 => 7
# );
# $fsm->OUTPUT_DEFAULTS(\%defaults);
#
# Set the state machine output widths (needed for calling Mux()):
# my %widths = (
#   output1 => 1,
#   output2 => 1,
#   output3 => 3
# );
# $fsm->OUTPUT_WIDTHS(\%widths);
#
# Add a state, with its name, transitions to other states and
# outputs:
# $fsm->add_state(
#   "start",                                   # new state name
#   [
#     {do_init => '0',},                       # inputs
#     "start",                                 # next state
#     {init_done => '0'},                      # outputs
#   ],
#   [
#     {do_init => '1'},                        # inputs
#     "init_state",                            # next state
#     {init_done => '0'},                      # outputs
#   ],
# );
# 
# 
# Add a chain of states, with a branch at the last state added.
# Possibly confusing fact: the second parameter, "number of states
# to add", doesn't count the terminal branch state.  So, the number
# of states added is actually one greater than the specified number.
# The reason for this is that often the final branch state is the
# interesting and necessary function being added, and 0 or more
# delays are required before it.
#
# $fsm->add_states(
#   "init_state",                      # new state
#   3,                                 # number of states to add
#   {init_done => '0'},                # outputs (all but last state)
#   [                                  # Define the last (branch) state
#     {read_n => '0'},                 # inputs
#     "do_read",                       # next state
#     {init_done => '1'},              # outputs
#   ],
#   [
#     {write_n => '0'},                # inputs
#     "do_write",                      # next state
#     {init_done => '1'},              # outputs
#   ],
#   [
#     {read_n => '0', write_n => '1'}, # inputs
#     "idle",                          # next state
#     {init_done => '1'},              # outputs
#   ],
# );
#
################################################################
my %fields = (
  TABLE => [],
  name => "unnamed_state_machine",
  OUTPUT_DEFAULTS => {},
  OUTPUT_WIDTHS => {},
  DEBUG => 0,
  start_state => undef,
  GENERATED => 0,
);

my %pointers = ();

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

# Debug-print feature: if DEBUG is true, print to STDERR.
################################################################################

=item I<dprint()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub dprint
{
  my $self = shift;
  
  $self->{DEBUG} && print STDERR @_;
}

# Convert a state name (from the state transition table) to its flipflop's
# Q output signal name.  Small, silly, and called from several places.
################################################################################

=item I<_create_flipflop_name()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _create_flipflop_name
{
  my ($self, $st) = @_;
  
  my $name = $self->{name} . "_$st";
  $self->dprint("_create_flipflop_name returns '$name'\n");
  return $name;
}

# add_state
# Add a single state, with 1 or more transitions to next states.
# Each transition contains input assignments (which select the
# transition to be taken), a next state (corresponding to the
# chosen trans, and outputs.
#
# Inputs:
# - a new state name
# - one or more list refs of the form:
#   [\%inputs, $next_state, \%outputs]
#
################################################################################

=item I<add_state()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_state
{
  my $self = shift;
  my $new_state = shift;

  $self->dprint(ref($self) . "::add_state('$new_state')\n");
  
  if (!defined($new_state))
  {
    ::VppError("e_fsm::add_state() usage error: no new state name.\n");
  }
  
  if (0 >= @_)
  {
    ::VppError("e_fsm::add_state() usage error: no next state info.\n");
  }
  
  while (@_)
  {
    # Copy the input list...
    my $listref = shift;
    my @list = @{$listref};
    
    # Convert the parameters to state transition table format, and add to
    # the table.
    if (3 != @list)
    {
      ::VppError("e_fsm::add_state(): incorrect number of elements in next-state info.\n");
    }
    
    my @compat_list = ($list[0], $new_state, $list[1], $list[2]);
    $self->dprint("\tadding cur: '$new_state'; next: '$list[1]'\n");
    push @{$self->{TABLE}}, \@compat_list;
  }
}

# add_states
# 
# Add 0 or more delay states followed by a branch state.
#
# Input:
# - a new state name
# - the number of states to add (in addition to the first)
# - an output value for all non-penultimate states
# - a branch condition for the final state, of the form:
#    [\%inputs, $next_state, \%outputs]
#
# Return value: a list of the state names which were added.
# 
################################################################################

=item I<add_states()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_states
{
  my $self = shift;
  my $new_state = shift;
  my $num_states = shift;
  my $output_ref = shift;
  my @added_state_list = ();

  if (!defined($new_state) || !defined($num_states) || !defined($output_ref))
  {
    ::VppError("e_fsm::add_states() usage error: no new state name.\n");
  }
  
  if (0 >= @_)
  {
    ::VppError("e_fsm::add_states() usage error: no next state info.\n");
  }
  
  my $state_base_name = $new_state;

  $self->dprint(ref($self) . "::add_states('$new_state')\n");
  
  $self->dprint("adding $num_states states starting with '$new_state'\n");
  
  my $nextstate = $state_base_name;
  for my $i (0 .. $num_states - 1)
  {
    # Form the names of the current and next states.
    my $curstate =  sprintf("$state_base_name\_%d", $i);
    if (0 == $i)
    {
      $curstate = $state_base_name;
    }
    $nextstate =  sprintf("$state_base_name\_%d", $i + 1);

    # Make a copy of the output hash for each new state.
    my %outputs = %{$output_ref};
    
    $self->add_state(
      $curstate,
      [
      {},
      "$nextstate",
      \%outputs,
      ]
    );
    
    push @added_state_list, $nextstate;
  }

  # Deal with last state, which may be a branch.
  $self->dprint("adding branch state.\n");
  $self->add_state(
    $nextstate,
    @_
  );

  push @added_state_list, $nextstate;
  
  # Return all the added states, or just the last added state
  # (scalar context).
  return wantarray ? @added_state_list : pop(@added_state_list);
}

# Set output value(s) for a particular state.
# Note: 
################################################################################

=item I<add_outputs()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_outputs
{
  my $self = shift;
  my ($state, $input_ref, $output_ref) = @_;
  
  my $success = 0;
  my %inputs = %{$input_ref};

  for my $i (0 .. -1 + @{$self->{TABLE}})
  {
    my $table_ref = $self->{TABLE};
    my $stt_ref = @{$table_ref}->[$i];
    my $curstate = @{$stt_ref}->[1];
    if (!defined($curstate))
    {
      print STDERR "failed to locate curstate at i = $i, state: '$state'\n";
    }
    
    if ($curstate eq $state)
    {
      my $existing_output_ref = @{$stt_ref}->[3];
      
      # Verify that this state transition meets the input requirement.
      my $stt_inputs_ref = @{$stt_ref}->[0];
      my %stt_inputs = %{$stt_inputs_ref};

      my $match = 1;
      for my $signal (keys %inputs)
      {
        if (exists($stt_inputs{$signal}))
        {
          if ($stt_inputs{$signal} ne $inputs{$signal})
          {
            $match = 0;
          }
        }
      }
      
      if ($match)
      {

        # Set all desired outputs in this state transition.
        for my $signal (keys %{$output_ref})
        {
          $existing_output_ref->{$signal} = $output_ref->{$signal};
        }
      }
    }
  }
}

################################################################################

=item I<get_next_state()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_next_state
{
  my $self = shift;
  my ($state, $input_ref) = @_;

  my %inputs = %{$input_ref};

  
  # TABLE is a list of references to 4-element lists of the form
  # ($input_hash_ref, $curstate, $nextstate, $output_hash_ref).
  
  # Search the entire table for elements which meet the requirement
  # $curstate == $state && $input_hash_ref satisfies $input_ref.
  # Remember that !exists(%{$input_hash_ref}->{$signal} implies
  # $signal is a don't care for that state transition.
  
  # If more than one next-state is found, it's an error.
  # Put matching states into a list, so that the unexpected event
  # can be reported.
  my @next_state = ();
  
  my @table = @{$self->{TABLE}};
  for my $i (0 .. -1 + @table)
  {
    my @stt = @{$table[$i]};
    
    ($stt[1] ne $state) and next;
    
    # Test each signal->value in $input_ref against
    # %{$stt[0]}.
    my %stt_inputs = %{$stt[0]};
    
    my $match = 1;
    for my $signal (keys %inputs)
    {
      if (exists($stt_inputs{$signal}))
      {
        if ($stt_inputs{$signal} ne $inputs{$signal})
        {
          $match = 0;
        }
      }
    }
    
    if ($match)
    {
      push @next_state, $stt[2];
    }
  }
  
  if (0 == @next_state)
  {
    return undef;
  }
  
  if (1 != @next_state)
  {
    ::VppError("Too many matches in get_next_state\n");
  }
  
  return $next_state[0];
}

################################################################################

=item I<update()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub update
{
  my ($self) = @_;

  # Do this only once!
  if (!$self->{GENERATED})
  {
    $self->{GENERATED} = 1;

    # Let's emit a one-hot state machine.  So... the first problem: state
    # assignment.  I just choose the most obvious mapping: each state has
    # numeric value i, where i is the order in which the state name appears
    # in the state table.

    # Here are a bunch of data structures to fill in with state table data.

    # State names -> numeric state value
    # s -> d where
    #   s: a state name
    #   d: that state's numeric value (non-negative integer).
    my %state_value = ();  

    # s -> [s_pred, i] where
    #   s: a state name
    #   s_pred: a predecessor state name
    #   i: inputs which cause transition from state s_pred to state s.
    my %state_predecessors = ();

    # s -> l where
    #   s: a state name
    #   l: the list of possible successor states to s
    my %state_successors = ();

    my $state_value = 0;

    # I think I can avoid warnings by explicitly making input and output
    # ports.
    my %inputs = ();
    my %outputs = ();
    
    # Scan the table and fill in a handful of data structures.
    for my $st_ref (@{$self->{TABLE}})
    {
      my $curstate = $st_ref->[1];
      my $nextstate = $st_ref->[2];

      my %input_assignments = %{$st_ref->[0]};
      my %output_assignments = %{$st_ref->[3]};

      # Copy the input and output signal names into my hashes.
      @{\%inputs}{keys %input_assignments} = values %input_assignments;
      @{\%outputs}{keys %output_assignments} = values %output_assignments;

      if (!exists($state_value{$curstate}))
      {
        $state_value{$curstate} = $state_value++;  
      }

      if (!exists($state_value{$nextstate}))
      {
        $state_value{$nextstate} = $state_value++;  
      }

      # Make a hash of predecessor states, by state.
      my @pred_list;
      if (!exists($state_predecessors{$nextstate}))
      {
        @pred_list = ();
      }
      else
      {
        @pred_list = @{$state_predecessors{$nextstate}};
      }

      # Add this predecessor state and a reference to its input hash.
      push @pred_list, [$curstate, $st_ref->[0]];
      $state_predecessors{$nextstate} = \@pred_list;

      # Make a hash of successor states, by state.
      # This is used as a sanity check later: states with no successors
      # are silly.
      my @succ_list;
      my $succ_list_ref;
      if (!exists($state_successors{$curstate}))
      {
        @succ_list = ();
      }
      else
      {
        $succ_list_ref = $state_successors{$curstate};
        @succ_list = @$succ_list_ref;
      }

      push @succ_list, $nextstate;
      $state_successors{$curstate} = \@succ_list;
      # End of grab-successor-states.
    }

    # print STDERR "inputs:\n", join "\n", keys %inputs, "\n";
    # print STDERR "outputs:\n", join "\n", keys %outputs, "\n";

    # Add ports for each input.
    for my $input_signal_name (keys %inputs)
    {
      
      $self->add_contents(
        e_port->new({
          name => $input_signal_name,
          direction => "input",
        })
      );
    }
    
    # Add ports for each output... but wait!  Check the
    # default output hash, for outputs which didn't appear
    # in the state transition table.

    @{\%outputs}{keys %{$self->{OUTPUT_DEFAULTS}}} =
      values %{$self->{OUTPUT_DEFAULTS}};
    for my $output_signal_name (keys %outputs)
    {
      $self->add_contents(
        e_port->new({
          name => $output_signal_name,
          direction => "output",
          width => $self->{OUTPUT_WIDTHS}->{$output_signal_name},
        })
      );
    }
      

    # Emit all reg declarations first, to avoid "!defined variable" error.
    # ::Vprint("\n // State registers.\n");
    # This is no longer necessary in europa.
    for my $st (sort {$state_value{$a} <=> $state_value{$b}} keys %state_value)
    {
      my $state_flipflop_Q = $self->_create_flipflop_name($st);

      # Register predeclarations
      $self->dprint(" reg $state_flipflop_Q;\n");
    }

    # for each state s
    #   for each predecessor state p
    #     s_d |= (state == p) && (inputs = p's inputs when it's s's predecessor)

    # my @current_contents = @{$self->{contents}};

    # Process states in numerical order.
    for my $st (sort {$state_value{$a} <=> $state_value{$b}} keys %state_value)
    {
      # $st is a state name; $state_value{$st} is its assigned state value.

      # Since this is a one-hot encoding, each state implies one flipflop.
      # I assign each flip-flop's output signal name here.
      my @pred_list = @{$state_predecessors{$st}};

      # Reality check: every state except the start state must have at least
      # one predecessor.
      if ((0 == @pred_list) and ($st ne $self->{start_state}))
      {
        ::VppError("state '$st' has no predecessors");
      }

      # Another check: states which have no successor states are senseless.
      my @succ_list = @{$state_successors{$st}};
      if (0 == @succ_list)
      {
        ::VppError("state '$st' has no successors");
      }
      # Each element of @pred_list is a list ref; each such list ref refers to 
      # a state value and its input hash.

      $self->dprint("\n // State '$st': $state_value{$st}\n");
      $self->dprint(" // predecessors:\n");

      # Each state FF's D input is a sum of products of the form:
      #   (state_foo == 1) & (input_1 == x1) & (input2 == x2) | (state_bar == 1) & (input_3 == x3)
      my @ff_d_terms = ();

      # Loop over all items in list of predecessors to this state.    
      my $comment = "";
      for my $pred_el_ref (@pred_list)
      {
        my @pred_el = @{$pred_el_ref};

        # All terms should contain a "state = blah" dependency.
        my $term = "(" . $self->_create_flipflop_name($pred_el[0]) . " == 1)";

        $self->dprint(" //   $pred_el[0]: (");

        # Grab the hash of input values which correspond to this transition.
        my %input_hash = %{$pred_el[1]};

        # Loop over all input variables and form a logic expression of their intersection.
        my @print_terms = ();
        for my $input_var (keys %input_hash)
        {
          $term .= " & ($input_var == $input_hash{$input_var})";
          push @print_terms, "$input_var == $input_hash{$input_var}";
        }

        $self->dprint(join " and ", @print_terms);
        $self->dprint(");\n");

        $term = "(" . $term . ")";        
        push @ff_d_terms, $term;
      }
      my $state_flipflop_Q = $self->_create_flipflop_name($st);
      my $input_expr = join " |\n", @ff_d_terms;

      my $this_state = e_register->new({
        comment => " Transitions into state '$st'.",
        name => $st,
        out => e_signal->new({name => $state_flipflop_Q, width => 1,}),
        in => $input_expr,

        # If this is the start state, then we want it to reset to the true
        # state.  All other flipflops should reset to 0.
        async_value => ($st eq $self->{start_state}) ? 1 : 0,
      });

      # Add this register to the module's contents.
      $self->add_contents($this_state);
      $self->dprint("added state $state_flipflop_Q to fsm\n");
    }

    # Ook.  Now it's time to specify the output equations.
    # for each output
    #   create a mux
    #   for each state which specifies a value for the output
    #     add one mux input with the corresponding output selected
    #     by the state bit and its inputs.

    # This is not very clever.  Some improvements:
    # - notice if an output is set to the same value in every state, and if
    #   so, set it to a constant.
    # - notice if an output is assigned a value in more than half of all states,
    #   and use that fact to reduce the number of mux inputs.
    # - notice if an output value is constant over all transitions from a
    #   state, and if so, simplify the logic expression (leonardo can't do it,
    #   as it has insufficient info to make the determination). -- done

    # o -> a list of elements of the form [s, v] where
    #   o: an output variable name
    #   s: a state with an assignment for o
    #   v: the value assigned to o in state s
    my %output_values = ();

    for my $st_ref (@{$self->{TABLE}})
    {
      my $curstate = $st_ref->[1];
      my $nextstate = $st_ref->[2];
      my %output_assignments = %{$st_ref->[3]};

      # Package up the output values in a more convenient form:
      #  indexed by output value name, look up the associated
      #  output value, state and input values.
      my $output_name;
      for $output_name (keys %output_assignments)
      {
        my @assignments = ();

        if (exists($output_values{$output_name}))
        {
          # This output already has assignments from some
          # other state.  Add to the list.
          @assignments = @{$output_values{$output_name}};
        }

        push @assignments,
          [$curstate,
          $output_assignments{$output_name},
          $st_ref->[0]];

        $output_values{$output_name} = \@assignments;
      }
    }

    # Run through the list of output values, creating a mux for
    # each one.
    for my $output_name (sort keys %output_values) 
    {
      my @mux_table = ();
      my @assignments = @{$output_values{$output_name}};

      $self->dprint("working on output '$output_name'\n");

      # Form the mux select expression, of the form
      # (<state> && (input1 == val1) && (input2 == val2) ... )

      # For this output, make a hash indexed by current state name,
      # containing a list of lists [iref, o], where iref is a reference to the hash of
      # input values of the transition, and o is the output.  This will be preprocessed
      # and then converted into an expression of the form
      #
      #   (<state> && (input1 == val1) && (input2 == val2) --> <output_value>
      #
      # for Mux().

      my %transition = ();
      for my $oa_ref (@assignments)
      {
        # Create the state flipflop, from the current state name.
        my $state_name = $self->_create_flipflop_name(@$oa_ref->[0]);
        my $state = @$oa_ref->[0];

        # Grab the hash of input values pertaining to this transition.
        my %input_hash = %{@$oa_ref->[2]};
        my $output_value = @$oa_ref->[1];

        if (!exists($transition{$state}))
        {
          $transition{$state} = [];
        }

        push @{$transition{$state}}, [\%input_hash, $output_value];
      }

      # Now, decide if any elements of the transitions list can be crushed down to a single entry.
      # For each state, if all transition output values are the same, and if all transitions from this
      # state (from the table) are mentioned, we can collapse to a single entry.
      for my $state (sort keys %transition)
      {
        # Grab a copy of the transitions from this state.
        my @list = grep {$state eq @{$_}->[1]} @{$self->{TABLE}};
        my $can_crush = 1;

        # It might be a good idea to verify that all input hashes from TABLE match the ones in @list.
        # But, since it's so much easier to do, I'm just going to compare numbers of transitions.  This
        # breaks if the e_fsm user defines redundant transitions in the TABLE, I think.

        my $last_output = undef;
        if ((0 + @list) == (0 + @{$transition{$state}}))
        {
          # All transitions are covered.  Ensure that all output values are the same.
          my $io_pair;
          for $io_pair (@{$transition{$state}})
          {
            my @list_io_pair = @{$io_pair};
            my %input_hash = %{$list_io_pair[0]};
            my $output_value = $list_io_pair[1];

            if (!defined($last_output))
            {
              $last_output = $output_value;
            }
            else
            {
              if ($last_output ne $output_value)
              {
                $can_crush = 0;
              }
            }
          }
        }
        else
        {
          $can_crush = 0;
        }

        if ($can_crush)
        {
          # If there's only one transition out of this state, no need to crush.
          if (1 != (0 + @{$transition{$state}}))
          {
            # Crush this transition down to an input-less output value.
            $self->dprint("crushing output '$output_name' in state '$state'\n");

            $transition{$state} = [];
            push @{$transition{$state}}, [{}, $last_output];
          }
        }
      }

      for my $state (sort keys %transition)
      {
        my $state_name = $self->_create_flipflop_name($state);

        $self->dprint("looking at state '$state_name'\n");

        for my $io_pair (@{$transition{$state}})
        {
          my @list_io_pair = @{$io_pair};
          my %input_hash = %{$list_io_pair[0]};
          my $output_value = $list_io_pair[1];

          my $this_expression = "($state_name";

          for my $input (sort keys %input_hash)
          {
            $this_expression .= " && ($input == $input_hash{$input})";
          }
          $this_expression .= ")";
          push @mux_table, $this_expression;

          $self->dprint("mux_table gets '$this_expression'\n");
          push @mux_table, $output_value;
          $self->dprint("mux_table gets '$output_value'\n");
        }
      }

      my $muxtype = "priority";
       
      # If there's no default, or the default is 0, make it an and-or mux.
      if (!exists($self->{OUTPUT_DEFAULTS}->{$output_name}))
      {
        $muxtype = "and-or";
      }
      else
      {
        my $default = $self->{OUTPUT_DEFAULTS}->{$output_name};
        
        $default =~ s/^[\d\']*[bodhBODH]//;
        
        if ($default !~ /[^0]+/)
        {
          $muxtype = "and-or";
        }
      }

      my $this_mux = e_mux->new({
        lhs => e_signal->new({
          name => $output_name,
          width => $self->{OUTPUT_WIDTHS}->{$output_name},
        }),
        type => $muxtype,
        table => \@mux_table,
      });
     
      # Set the default if it exists. Oh, but: and-or muxes suddenly hate me
      # if I give them a default, even if it's 0.  For and-or muxes, rely on
      # the implicit '0' default.
      if (($muxtype ne "and-or") and
        exists($self->{OUTPUT_DEFAULTS}->{$output_name}))
      {
        $this_mux->default($self->{OUTPUT_DEFAULTS}->{$output_name});
      }

      $self->add_contents($this_mux);

      # Delete the just-created output from the default output list.
      # Later, we'll notice which outputs are only ever assigned their
      # default value (therefore no longer present in %output_values) and
      # do a simple assignment for them.
      delete $self->{OUTPUT_DEFAULTS}->{$output_name};
    }

    # Do a simple assignment for any remaining outputs; these are the
    # ones which didn't get an assignment in the state table.
    for my $output_name (keys %{$self->{OUTPUT_DEFAULTS}})
    {
      # Simple assignment.
      $self->dprint(" assign $output_name = $self->{OUTPUT_DEFAULTS}->{$output_name};\n");
      my $this_assignment = e_assign->new({
        comment => " Output $output_name has a constant value.",
        lhs => e_signal->new(["$output_name"]),
        rhs => "$self->{OUTPUT_DEFAULTS}->{$output_name}",
      });

      $self->add_contents($this_assignment);
    }
  }
  
  # Go off and call the base class update.
  $self->SUPER::update();
}

qq{
The time has come to say goodnight, 
My how time does fly. 
We've had a laugh, perhaps a tear, 
and now we hear goodbye. 

I really hate to say goodnight, 
for times like these are few. 
I wish you love and happiness, 
In everthing you do. 

The time has come to say goodnight, 
I hope I've made a friend. 
And so we'll say "May God bless you," 
Until we meet again. 

~~ Red Skelton ~~ 
};



=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_module

=begin html

<A HREF="e_module.html">e_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
