package e_fifo_with_registered_outputs;
use europa_utils;
@ISA = ("e_instance");
use strict;

#This FIFO is composed of as many stages as the parameterized depth. Stage0 is
#the FIFO output. When data progresses from a higher stage, to a lower stage,
#then it's getting closer to prime time. When data moves from a lower stage to
#a higher stage, then we are being written into and not read. When both a read
#and a write happens, the incoming data goes to the highest full stage, and the
#controls bits (full_XX to full_0) don't change.

#There are two sets of muxes and registers. One of these sets contains the data,
#and are unsurprisingly called data_xx. The other set contains the control
#information, to keep track which stage has valid (is full) data.

my %fields = ( depth => 4 );

my %pointers = ();

&package_setup_fields_and_pointers( __PACKAGE__, \%fields, \%pointers, );

sub new {
 my $class = shift;
 my $this  = $class->SUPER::new(@_);
 $this->setup_module;
 return $this;
}

sub setup_module {
 my $this = shift;

 my $name = $this->name() || ribbit("Please setup a name for the module");
 my $mod = e_module->new( { name => $name . "_module" } );

 my $depth = $this->depth();
 my $width = 8;

 my @width_conduit_array = ( "data_in", "data_out" );

 e_assign->new( [ "data_out", "stage_0" ] )->within($mod);
 e_assign->new( [ "full",     "full_" . ( $depth - 1 ) ] )->within($mod);
 e_assign->new( [ "empty",    "!full_0" ] )->within($mod);

 # We create one extra constant signal to avoid an exception in the loop.
 e_assign->new( [ "full_$depth", 0 ] )->within($mod);

 for ( my $stage = $depth - 1 ; $stage >= 0 ; $stage-- ) {
  my $higher_stage;
  if ( $stage < $depth - 1 ) {
   $higher_stage = "stage_" . ( $stage + 1 );
  }
  else {
   $higher_stage = "data_in";
  }
  my $out      = "stage_$stage";
  my $data_mux = e_mux->new(
   {
    name => "data_$stage",
    type => "selecto",

    # Selecto in the highest stage will point
    # to a constant signal and Quartus will
    # eat the extra logic.
    selecto => "full_" . ( $stage + 1 ) . " & ~clear_fifo",
    table => [
     0 => "data_in",
     1 => $higher_stage,
    ],
    out      => "p$stage\_$out",
   }
  );
  
  my $lower_control_bit = "full_" . ($stage + 1);
  my $data_reg = e_register->new(
   {
    name => "data_reg_$stage",
    in => "p$stage\_$out",
    out => "$out",
    sync_reset => "sync_reset & full_$stage & !(($lower_control_bit == 0) & read & write)",
    enable => "clear_fifo | sync_reset | read | (write & !full_$stage)",
    reset => '',
   }
  );
    
  push( @width_conduit_array, $out );
  $mod->add_contents($data_mux);
  $mod->add_contents($data_reg);
  
  my $lower_control_stage;
  my $higher_control_stage;
  if ( $stage != 0 ) {
   $higher_control_stage = "full_" . ( $stage - 1 );
  }
  else {
   $higher_control_stage = 1;
  }
  if ( $stage < $depth - 1 ) {
   $lower_control_stage = "full_" . ( $stage + 1 );
  }
  else {
   $lower_control_stage = 0;
  }

  my $control_mux = e_mux->new(
   {
    name    => "control_$stage",
    type    => "selecto",
    selecto => "(read & !write)",
    table   => [
     0 => $higher_control_stage,
     1 => $lower_control_stage,
    ],
    out => "p$stage\_full_$stage",
   }
  );
 
  my $clear_fifo = "clear_fifo";
  if($stage == 0)
  {
    $clear_fifo = "clear_fifo & ~write";
  }

  my $control_reg = e_register->new(
   {
    name       => "control_reg_$stage",
    in         => "p$stage\_full_$stage",
    out        => "full_$stage",
    enable     => "clear_fifo | (read ^ write) | (write & !full_0)",
    sync_reset => $clear_fifo,
   }
  );
  # If we are reading or writing, we want to enable the muxes. In the case
  # that both signals are active, we want to keep the control signals
  # constant. Now, this breaks down when both read and write are active
  # but our FIFO was originally empty, hence we want to enable if
  # full_0 == 0 and they want to write.

  $mod->add_contents($control_mux);
  $mod->add_contents($control_reg);
 }
 
 my $how_many_ones = "how_many_ones";
 my $one_count_plus_one = "one_count_plus_one";
 my $one_count_minus_one = "one_count_minus_one";
 my $updated_one_count = "updated_one_count";

 e_signal->new({ name => $how_many_ones, width => (Bits_To_Encode($depth) + 1)})
               ->within($mod);

 e_assign->new([$one_count_plus_one, "$how_many_ones + 1"])->within($mod);
 e_assign->new([$one_count_minus_one, "$how_many_ones - 1"])->within($mod);

 e_width_conduit->new( [
                         $how_many_ones,
                         $one_count_plus_one,
                         $one_count_minus_one,
                         $updated_one_count
                       ]
		       )->within($mod);
          
 e_mux->new(
  {
    name      => "updated_one_count",
    table     => [
                   "((clear_fifo | sync_reset) & !write)" => "0",
                   "((clear_fifo | sync_reset) & write)" => "|data_in",
                   "read & (|data_in) & write & (|stage_0)"
                               => $how_many_ones,
                   "write & (|data_in)"  => $one_count_plus_one,
                   "read & (|stage_0)"  => $one_count_minus_one
                 ],
    default   => $how_many_ones,
    out       => $updated_one_count,
  }
 )->within($mod);
                   					
 e_register->new(
  {
   name        => "counts how many ones in the data pipeline",
   in          => $updated_one_count,
   out         => $how_many_ones,
   enable      => "clear_fifo | sync_reset | read | write",
  }
 )->within($mod);

 e_register->new(
  {
   name        => "this fifo contains ones in the data pipeline",
   in          => "~(|$updated_one_count)",
   out         => "fifo_contains_ones_n",
   enable      => "clear_fifo | sync_reset | read | write",
   async_value => "1",
  }
 )->within($mod);

 e_width_conduit->new( \@width_conduit_array )->within($mod);
 $this->module($mod);
}

return 1;
