



################################################################
# CLASS: e_default_module_marker
#
#  In general, every time you create a design element
#   (i.e. an "e_thing_that_can_go_in_a_module"), you have to 
#  somehow explicitly add it to some e_module object.
#
#  This tends to rub the experienced HDL coder the wrong way.
#  The experienced coder is used to saying:
#
#      module foo
#         ..various contents of foo...
#      endmodule
#
#  People get used to the idea that when you create an
#  assignment, for example, it goes into the current module.
#  In other words, some people like doing this:
# 
#    A:       e_assign->new (["result", "x & ~y"]);    
#
#  And dislike doing this:
#
#    B:       e_assign->new ({within => $current_e_module_object,
#                             lhs    => "result",
#                             rhs    => "x & ~y",
#                            });
#
#
#  Problem is, of course, that in case (A), we have created a
#  new e_assign object and then dropped it on the floor.  In case
#  (B) the object isn't dropped because we specify which module
#  it should go in.
#
#  The people who like doing (A) would, of course, like the 
#  fine e_assign object they've created to go in the CURRENT module.
#  You know--the module they're building.  The module under construction.
#  The module they're working on.  Capice?
#
#  This here object is the mechanism by which europa can "capice" your
#  notional current module.  You create one of these objects to 
#  mark a (preexisting, properly-constructed) e_module object as 
#  being THE current module.  
#
#  Why not just set some SCARY_LOOKING_GLOBAL to do the same thing?
#  Here's why:  Because then you'd have to remember to un-set it when
#  you were done, or the whole downstream world would hate you.
#
#  By marking THE current module using an object, we can automate the 
#  process of turning off the lights when you leave the room.  
#  Here's how it works.
# 
#  Suppose you've written a Perl subroutine which builds your current
#  module (many people like to do this.  They're the same kind of people 
#  who like to do (A).  OK, it's me--alright?  you caught me.).
#  Suppose you've created a module, and you want all your stuff to 
#  go into it without having to say "within => " all the time.
#  Here's how you'd do it:
#  
#     sub build_my_module
#     {
#        my $pony_module = e_module->new ({name => "my_little_pony"});
#
#        my $marker = e_default_module_marker->new ($pony_module);
#
#        e_assign->new(...);
#        e_register->new(...);
#          ... many things that go in modules are created...
#        e_mux->new (...);
#
#        return $working_module;    # Done.  Pass-back result.
#     }
#
# By creating the object "$marker", you "marked" the e_module object
# "$pony_module" as being THE current object.  All the stuff you create
# from here on out goes into $pony_module unless you say otherwise--
# UNTIL $marker IS DESTROYED (goes out-of-scope).
# 
# That's why we bother with all this object foolishness (instead of 
# just setting a SCARY_LOOKING_GLOBAL).  The current module is set
# when you create the marker, and un-set when the marker goes away.
# How very polite.  As an added bonus, this object can (internally)
# remember the previous current-module (if any), and restore it when
# you're done.  How very, very polite.
#
# Of course, deep down inside, this whole thing works using a 
# SCARY_LOOKING_GLOBAL.
#
################################################################


=head1 NAME

e_default_module_marker - description of the module goes here ...

=head1 SYNOPSIS

The e_default_module_marker class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_default_module_marker;

use e_module;
@ISA = ("e_object");
use e_default_module_marker;

use strict;
use europa_utils;

# Where the current module is kept.
my $GLOBAL_CURRENT_MODULE = e_module->dummy();

#I'm paranoid, so I'm going to give each one of these things
# a serial number and make sure they go away in the order they
# were created.
my $serial_number_counter = 0;

################################################################
# e_default_module_marker::new
#
# Constructor takes one argument: an e_module object (reference).
################################################################
my %fields = (
              serial_number => 0,
              );

my %pointers = (
                previous_module => e_module->dummy(),
               );

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

################################################################################

=item I<new()>

Object constructor

=cut

sub new {
  my $this  = shift;
  my $self = $this->SUPER::new();

  $self->current_module (@_);
  $self->serial_number ($serial_number_counter++);

  return $self;
}


################################################################
# current_module()
#
# Sets-or-gets the current module
#
# This is not just an AUTOLOAD-buit access function because 
# our notion of the "current module" is stored in that nasty 
# global variable.
#
################################################################
################################################################################

=item I<current_module()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub current_module
{
  my $this = shift;
  my $marked_module = shift;
  &ribbit ("Too many arguments") if @_;

  if ($marked_module) {
    &ribbit ("Must be constructed from an e_module") 
      unless (&is_blessed ($marked_module) && $marked_module->isa("e_module"));

    $this->previous_module($GLOBAL_CURRENT_MODULE);
    $GLOBAL_CURRENT_MODULE = $marked_module;

  }

  return $GLOBAL_CURRENT_MODULE;
}

################################################################
# add_contents()
#
# Adds an "e_thing" (or whatever) to the global-current-default
# object.  This function should only ever be called when a
# new "e_thing_that_can_go_in_a_module" is created, by the 
# "e_thing_that_can_go_in_a_module" constructor.
#
# Does nothing, of course, if there is no global current module.
################################################################
################################################################################

=item I<add_contents()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_contents
{
  my $this = shift;
  return () if $GLOBAL_CURRENT_MODULE->isa_dummy();

  foreach my $thing (@_) {
    # Check for suspicious persons:
    &ribbit ("expected a thing that can go in a module not ($thing)") 
      unless (&is_blessed ($thing) && 
              $thing->isa ("e_thing_that_can_go_in_a_module"));

    $thing->within ($GLOBAL_CURRENT_MODULE);
  }
  return $GLOBAL_CURRENT_MODULE->contents();
}

################################################################################

=item I<DESTROY()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub DESTROY
{
  my $this = shift;

  # Make sure that the marker being destroyed is, identically, 
  # the most-recent one we created:
  #
  &ribbit ("default module-markers being destroyed in unexpected order")
    if $this->serial_number() != --$serial_number_counter;

  $GLOBAL_CURRENT_MODULE = $this->previous_module();
}


1;  #One!  One vonderful pachage!  Ah, ah, ah!




=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_object

=begin html

<A HREF="e_object.html">e_object</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
