






################################################################
# CLASS: e_cpu_fifo
#
################################################################


=head1 NAME

e_cpu_fifo - description of the module goes here ...

=head1 SYNOPSIS

The e_cpu_fifo class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_cpu_fifo;

@ISA = qw(e_module);

use e_module;
use e_port;
use e_parameter;
use europa_utils;

use strict;
my %all_unique_names = ();

my %fields = (
  name_stub    => "",
  data_width   => 1,
  fifo_depth   => 1,
  use_flush    => 0,
);

my %pointers = (
  unique_names => \%all_unique_names,
);

&package_setup_fields_and_pointers(__PACKAGE__,
                                   \%fields,
                                   \%pointers);

################################################################################

=item I<new()>

Object constructor

=cut

sub new
{
  my $this = shift;
  my $self = $this->SUPER::new(@_);

  # Go off and make the ports, etc.
  $self->make_fifo();
    
  return $self;
}

################################################################################

=item I<make_fifo()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub make_fifo
{
  my $this = shift;

  my $name = $this->name_stub() . "_fifo_module";
  while ($this->{unique_names}->{$name})
  {
    $name =~ s/_(\d+)$//;
    my $digit = $1 || 0;
    $digit++;
    
    $name .= "_" . $digit;
  }
  $this->{unique_names}->{$name} = 1;

  $this->name($name);
  

  # Fifo depth need not be a power of 2 anymore.
  my $depth = $this->fifo_depth();
  my $fifo_address_bits=($depth > 0)? (&Bits_To_Encode($depth-1)) : 0; 

#print "fifo depth is $depth\n";
#print "fifo address_bits is $fifo_address_bits\n";

  # The fifo read data output has to be exported explicitly,
  # since it feeds an e_ram (blind instance).
  # $this->add_contents(
  #   e_signal->new({
  #     name => "fifo_rd_data",
  #     width => $this->data_width(),
  #     export => 1,
  #   }),
  # );

  # Ports.
  $this->add_contents(
    e_port->new(["clk"]),
    e_port->new(["clk_en"]),
    e_port->new(["reset_n"]),
    e_port->new(["fifo_write"]),
    e_port->new(["fifo_read"]),
    e_port->new(["fifo_wr_data", $this->data_width()]),
    e_port->new(["fifo_rd_data", $this->data_width(), "output"]),
#    e_port->new(["fifo_full",    1,                   "output"]),
    e_port->new(["ic_read",       1,                   "output"]),
    e_port->new(["fifo_read_data_bad", 1,              "output"]),
  );

  # determine if we use a 'flush' signal, or if it should be set to 0
  if ($this->use_flush) {
    $this->add_contents( e_port->new(["flush",        1]) );
  } else {
    $this->add_contents(
        e_assign->new({
            lhs => e_signal->new(["flush", 1]),
            rhs => 0,
        }),
    );
  }

  # Internal FIFO control signals.

  $this->add_contents(
    # if the fifo is empty, let is be empty until there is a write without
    # a (good) read.
    # if the fifo is not empty, let it remain so until there is a read
    # (and no write, but this is taken care of by the "priority") that
    # will set the read pointer and write pointer equal.  Or until all
    # data is flushed out. 
    # If there is a flush, any laten
    e_register->new({
        q   => e_signal->new(["internal_fifo_empty"]),
        sync_set    => "fifo_becoming_empty",
        sync_reset  => "(fifo_write & ~fifo_read)",# & internal_fifo_empty 
        priority    => "set", 
        async_set   => "reset_n",
        async_value => "1", 
    }),

    # ic_read is almost the same as empty, except that we have to promise
    # not to change it when "wait" is true.
    # To catch the sync_set if we miss it due to a wait, just look at whether
    #   the fifo is empty (easy!).
    # But if we miss the reset, we need a more complicated method to remember
    #   (see dont_forget_to_reset_ic_read below).
    #
    e_register->new({
        q   => e_signal->new(["ic_read"]),
        sync_set    => "set_ic_read",
        sync_reset  => "reset_ic_read",
        priority    => "reset", 
        enable      => "~ic_wait",
        async_set   => "reset_n",
        async_value => "0", 
    }),
    e_assign->news (
      [["set_ic_read", 1, 0, 1],  "fifo_becoming_empty | internal_fifo_empty"],
      [["reset_ic_read", 1, 0, 1], "(fifo_write & ~fifo_read        ) || ".
                                   "  (dont_forget_to_reset_ic_read &&   ".
                                   "    ~internal_fifo_empty           ) "],
      [["ic_read_confusion", 1, 0, 1],  "set_ic_read & reset_ic_read"],
    ),
    # On the priority of setting/resetting ic_read:
    # It may happen that the CPU is getting back data 
    #
    # It can (occasionally) happen that internal_fifo_empty gets a reset-cue at
    # some point when ic_read can't respond.  This condition must be
    # remembered, which is what "dont_forget_to_reset_ic_read" is all about.
    # This bit is set when:
    #   - ic_wait is true and ic_read is therefore disabled
    #   AND
    #   - there was a write to the fifo without a read, indicating that
    #   there is no longer an empty fifo, and therefore no more reason to
    #   read.
    # We don't want to miss our reset-cue, so, we'll set this bit.  
    # Note that the ~ic_wait is implicit in the sync_set term.
    # Resetting this bit means "You may continue to read or not read as
    # you see fit."
    # This bit is reset for one of two reasons:
    #   - ic_read is done waiting, and we have successfully reset it. 
    #   OR
    #   - while ic_read still waiting, the fifo becomes empty again, thus
    #   obliterating our reason to reset ic_read.
    #
    # TPA :
    # All that is true.  But guess what:  Adding that new term 
    # to the logic makes this very register the critical path for the 
    # whole CPU.  And all other paths are over 80MHz.  What bad luck.  
    # Clearly, we have to find another, faster way of doing this same
    # job.   Here's what I'm thinking:
    #    We reset this flip-flop if we ever notice that it is set
    #    (which comes directly from a register--itself) and that 
    #    the fifo is currently empty (which also comes directly 
    #    from a register).  The only way this can ever arise is if 
    #    "fifo_becoming_empty" went true (which is why 
    #    "internal_fifo_emtpy" is true) and if ic_wait was true
    #    last time (which is why this signal is still true this time).
    #    
    #    In short, I claim adding the reset-term:
    #       (internal_fifo_empty && dont_forget_to_reset_ic_read)
    #    Is mathematically equivalent to, but faster than, adding the term:
    #       (fifo_becoming_empty && ic_wait).
    #                           
    # We shall see.
    e_register->new ({
        q           => "dont_forget_to_reset_ic_read",
        sync_set    => "(fifo_write & ~fifo_read) && ic_wait",
        sync_reset  => "
                (dont_forget_to_reset_ic_read && ~ic_wait             ) ||
                (internal_fifo_empty && dont_forget_to_reset_ic_read )  ",
                # This was Peter's hard-won new term:
                # || (fifo_becoming_empty && ic_wait)",  
        priority    => "reset",
        enable      => "1'b1",
        async_set   => "reset_n",
        async_value => "1'b1",
    }),
  );

  # FIFO control signals dependent upon depth:
  if ($depth > 1) {
    $this->add_contents(
        # Some handy signals: is the number of items in the fifo
        # about to increment, decrement or remain the same?
        e_assign->new({
            lhs => e_signal->new(["fifo_inc"]),
            rhs => "fifo_write"
        }),
        e_assign->new({
            lhs => e_signal->new(["fifo_dec"]),
            rhs => "fifo_read & ~(fifo_read_data_bad)"
        }),

        # Write-address. circular, 1-hot register addresser.    
        # Note that when there's a flush, the read pointer moves to be
        # equal with the write pointer, whatever it may be.  This ensures that
        # if there is a write with a flush, the write will not be forgotten.
        e_shift_register->new ({
            name               => "rdaddress_calculator",
            direction          => "MSB-first",
            parallel_out       => e_signal->new (["read_pointer", $depth]),
            serial_in          => "read_pointer\[($depth-1)\]",
            shift_enable       => "fifo_dec",
            shift_length       => $depth,
            async_value        => 1,
            parallel_in        => "1",
            load               => "flush",
        }),

        # Read-address. circular, 1-hot register addresser.    
        e_shift_register->new ({
            name               => "wraddress_calculator",
            direction          => "MSB-first",
            parallel_out       => e_signal->new (["write_pointer", $depth]),
            serial_in          => "write_pointer\[($depth-1)\]",
            shift_enable       => "fifo_inc",
            shift_length       => $depth,
            async_value        => "1",
            parallel_in        => "1",
            load               => "flush",
        }),

        #e_assign->new({
        #    lhs => e_signal->new(["internal_fifo_empty"]),
        #    rhs => "(write_pointer == read_pointer)",
        #}),
        e_assign->new({
            lhs => e_signal->new(["next_read_pointer", $depth]),
            rhs => "{read_pointer[($depth-2):0], read_pointer[($depth-1)]}",
        }),

        e_assign->new({
            lhs => e_signal->new(["fifo_becoming_empty", 1]),
            rhs => "
                ((next_read_pointer==write_pointer) 
                    & (fifo_read & ~fifo_write ) )
                | flush", 

        }),

        e_assign->new ({
           lhs => e_signal->new ({name         => "ic_read_prime",
                                  width        => 1,
                                  never_export => 1,            }),
           rhs => "internal_fifo_empty || continue_read_cycle",
        }),
        
        e_register->new ({
           q          => [continue_read_cycle => 1],
           sync_set   => "ic_read_prime",
           sync_reset => "~ic_wait",
           priority   => "reset",
           enable     => "1'b1",
        }),
              
        e_assign->new({
            lhs => "fifo_read_data_bad",
            #rhs => "(internal_fifo_empty & ~fifo_write)",
            rhs => "(internal_fifo_empty & ~(fifo_write)) | flush",
        }),

        e_assign->new ({
           lhs => e_signal->new ({name         => "bad_news",
                                  width        => 1,
                                  never_export => 1,          }),
           rhs => "ic_read_prime ^ ic_read",
        }),
    );
  } elsif ($depth>0) { # depth==1
    $this->add_contents( 
        e_assign->new ({
            lhs => e_signal->new({
                name => "write_pointer",
                width => 1, }),
            rhs => "1",
        }),
        e_assign->new ({
            lhs => e_signal->new({
                name => "read_pointer",
                width => 1, }),
            rhs => "1",
        }),
        # internal_fifo_empty is also ~internal_fifo_full.  
        # a read without a write means we're empty.  
        # we're only full when there's a write and no read.
        # If there is both a read and a write, then internal_fifo_empty should
        # keep whatever its old value was.
        # See internal_fifo_full logic above. 
        e_assign->new({
            comment => "set to internal_fifo_empty",
            lhs => "fifo_becoming_empty",
            rhs => "(fifo_read & ~fifo_write) | flush", 
        }),
        # NOTE : possible bug: this differs from the above case.  I don't see
        # why, at this point, they should be different. 
        e_assign->new({
            lhs => "fifo_read_data_bad",
            rhs => "(internal_fifo_empty & ~fifo_write)",
        }),
    );
  } else {  # depth == 0 
    &ribbit ("Nios 2.0 does not support a non-existant cpu fifo.");
    $this->add_contents( 
        e_assign->new({
            lhs => e_signal->new(["internal_fifo_empty"]),
            rhs => "1'b1",
        }),
        e_assign->new({
            lhs => "fifo_read_data_bad",
            rhs => "~fifo_write",
        }),

        # a read transaction, once set in motion, MUST continue until the
        # transaction is completed.  This bit allows us to keep track of 
        # the fact that we're in a read cycle, even if our reason for 
        # starting a read goes away.
        #
        e_register->new ({
           q           => [continue_read_cycle => 1],
           sync_set    => "ic_read",
           sync_reset  => "~ic_wait",
           priority    => "reset",
           enable      => "1'b1",
        }),
        e_assign->new({
            lhs => "fifo_empty",
            rhs => "fifo_read",
        }),
        e_assign->new({
            lhs => "ic_read",
            rhs => "fifo_empty || continue_read_cycle",
        }),
    );
  }

  # Fifo empty/full logic.
 
  
  # Tell the world the fifo is empty if it actually is empty.
  # But, if a fifo_write is happening, lie, so that someone
  # might be inspired to read.  If that happens, it's a pass-thru,
  # and doesn't get written into the FIFO memory, nor does the
  # read get counted against the FIFO memory (which is, after all,
  # empty).  See below qualifications of fifo_write and
  # fifo_read
  
  
  # Finally, instantiate the fifo's storage units and wire it up.
  if (0) {  # make the fifo ram based.
    if (($depth | ($depth - 1)) != ($depth * 2 - 1))
    {
        ribbit("fifo depth '$depth' is not an integer power of 2!");
    }
    $this->add_contents(

        e_signal->new({
            name => "wraddress",
            width => $fifo_address_bits,
        }),
        e_process->new({
            asynchronous_contents => [
                e_assign->new({
                    lhs => "wraddress",
                    rhs => 0,
                }),
            ],
            contents => [
                e_if->new({
                    condition => "flush", 
                    then => [
                        e_assign->new({
                            lhs => "wraddress",
                            rhs => "0",
                        }),
                    ],
                    else => [ 
                        e_if->new({
                            condition => "fifo_inc", 
                            then => [
                                e_assign->new({
                                    lhs => "wraddress",
                                    rhs => "wraddress - 1",
                                }),
                            ],
                        }),
                    ], # end of else
                }),
            ], # end of e_process contents
        }),

        e_signal->new({
            name => "rdaddress",
            width => $fifo_address_bits,
        }),
        e_process->new({
            asynchronous_contents => [
                e_assign->new({
                    lhs => "rdaddress",
                    rhs => 0,
                }),
            ],
            contents => [
                e_if->new({
                    condition => "flush",
                    then => [
                        e_assign->new({
                            lhs => "rdaddress",
                            rhs => "0",
                        }),
                    ],
                    else => [
                        e_if->new({
                            condition => "fifo_dec",
                            then => [
                                e_assign->new({
                                    lhs => "rdaddress",
                                    rhs => "rdaddress - 1",
                                }),
                            ],
                        }),
                    ],
                }),
            ],
        }),
        e_ram->new({
            name        => $this->name() . "_fifo_ram", 
            port_map    => {
                wren    => "fifo_write",
                data    => "fifo_wr_data",
                q       => "internal_fifo_rd_data",
                wrclock   => "clk"  ,
                wraddress => "wraddress",
                rdaddress => "rdaddress",
            },
        }),
    );
  } else { # make the fifo be a set of registers
    my @rd_mux_table;
    for (my $i=0; $i<$depth ; $i++) {
        my $select_bit = $i;
        my $reg_name = "fifo_reg_". $i;
        my $reg_write_select = $reg_name . "_write_select" ;
        my $reg_read_select = $reg_name . "_read_select" ;
        $this->add_contents(
            e_register->new({
                q   => e_signal->new({
                    name    => $reg_name, 
                    width   => $this->data_width()}),
                d   => "fifo_wr_data",
                enable  => "$reg_write_select & fifo_write",
            }),
            e_assign->new({
                lhs => e_signal->new([$reg_write_select, 1]),
                rhs => "write_pointer [$select_bit]",
            }),
            e_assign->new({
                lhs => e_signal->new([$reg_read_select, 1]),
                rhs => "read_pointer  [$select_bit]",
            }),
        );
        push (@rd_mux_table, ($reg_read_select  => $reg_name));
    }
    if ($depth>0) {
        $this->add_contents(
            e_mux->new ({
                out   => "internal_fifo_rd_data", 
                table => [ @rd_mux_table ], 
            }),
        );
    } else {
        $this->add_contents(
            e_assign->new({
                lhs => "internal_fifo_rd_data", 
                rhs => "0",
            }),
        );
    }
  }
    
  # For pass-thru, mux the fifo memory output and its input
  # out to the read-data output.  When pass-through is active,
  # the data being "written" just goes right out.
  $this->add_contents(
    e_signal->new({
      name => "internal_fifo_rd_data",
      width => $this->data_width(),
    }),    
    e_assign->new({
      lhs => "fifo_rd_data",
      rhs => "internal_fifo_empty ? fifo_wr_data : internal_fifo_rd_data",
    }),
  );


}

1;

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_module

=begin html

<A HREF="e_module.html">e_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
