






################################################################
# e_control_bit.pm
# 
# What is a control bit?
#
# Well, as far as the module that you add it to is concerned, it's
# nothing but a lil' old port.  
#
# But, secretly, when the lights all go down, it's.. Oh, so much more.
#
################################################################


=head1 NAME

e_control_bit - description of the module goes here ...

=head1 SYNOPSIS

The e_control_bit class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_control_bit;
use europa_utils;
use e_pipe_module;
use e_mnemonic;
use e_port;
use e_initial_block;
use e_readmem;
use format_conversion_utils;
@ISA = ("e_port");
use strict;

# A global static database of every control bit anyone ever created.
my %all_control_bits_by_name = ();
my @_rommed_control_bits = ();   # Ordered-list of rommed control-bits.
my @_rom_contents = ();          # And the corresponding rom contents.

################################################################
# new
# 
# There are two ways you can call the e_control_bit constructor.
# 
# **** From A String.
# 
# First, you can just pass-in a plain-old string.  Just a string.
# not a reference to a list, the first element of which is a string.
# No, no no.  Just a string.  
#
# If you pass a string to this constructor, it is taken as the name
# of your control bit (and, thence, the input-port on your module).
# And the name must internally encode the name of the mnemonics
# associated with this control bit.  The name MUST start with the
# following character sequence, or else it is an ERROR:
#
#                do_i
#
# If your string does'nt start with a lowercase 'd' and a lowercase
# 'o' and an underscore, followed by a lowercase 'i', then your
# program will terminate with an error. Have I made myself clear?
# Good.  Now then.  The rest of your string "describes" the opcodes
# that set your bit.  
#
# I hear you saying "What d'yall mean by 'describes', you pansy?"  
# What I mean is:  the rest of your string is taken as a very 
# primitive regexp.  This expression is matched against all 
# known mnemonics.  It must match one or more mnemonics.  
# Your bit (input-port) will be set TRUE whenever one of those 
# matched mnemonics (instructions) passes through your pipe-stage
# (wherever you are.. more on that later.).  Now, you are 
# probably wondering: "What kind of 'primitive regexp' have you
# wierdos cooked-up?"  I'll tell you what kind.  It would have been 
# nice to just give your control-bits names that have regexps
# embedded in them, like this:
# 
#     e_control_bit->new ("do_i/^ADD(.?)/");
#
# But there are obvious problems with this--not least, neither Verilog
# nor VHDL will be happy with this port name, what with all the
# slashes and up-arrows and such.  So, instead, we invent our own 
# hdl-legal wildcard character: 'x' (lowercase eks).    This wildcard
# is a bit funny.  It has the following meaning:  It matches 
# ONE OR MORE character when it appears in the middle of your
# description, and it matches ANY NUMBER, INCLUDING ZERO characters 
# when it appears at the end of your description.  Here an example 
# might help:
#
#     i_doRESTORE   ; Input goes true ONLY for RESTORE instructions.
#     i_doADDx      ; Input goes true for all of: ADD, ADDC, and ADDI.
#     i_doRxC       ; Input goes true for both of: RLC, RRC;
#     i_doEXTxd     ; Input goes true for both of: EXT8d, EXT16d
#
# **** Constructing from a two-element list: [name, list_ref(mnemonics)]
#
# Now then.  What if you want to specify a control-bit that goes
# true for some disparate grab-bag of instructions that don't map 
# into a nice, regular expression? Consider this example torn from
# today's headlines:
#
#  e_control_bit->new 
#        (["do_zero_a",   qq(MOV MOVI BR BSR JMP CALL 
#                            TRAP TRET BGEN NOT NEG ABS RDCTL)   ]);
#
# Here we see someone trying to construct a control-bit named
# "do_zero_a", which is set for a disparate grab-bag of mnemonics
# (try making an "x" for those suckers).  Note that "do_zero_a"
# does not start with "do_i".  In fact, the "do_i" rule, draconian 
# though it may be, only applies to control-bits made with the
# string-only constructor.  For the two-element-list constructor,
# you can name the control bit any damned thing you want.
#
# Now that you've seen this version of the constructor in action,
# it's not too hard to see what it does.  The first argument is 
# the name, and the second thing is a reference to a list of all 
# our associated mnemonics (including the same x-regexp expansion, 
# per the list above.
#
# **** Reference Previous Control Bits.
#      
# Sometimes you need a control bit in, say, pipe-stage 1, but then
# you need the same bit again later for a different purpose, (but
# set by the same set of instructions) in pipe-stage 3.  It would be a 
# shame and a big waste to generate the logic to compute this same
# bit twice, when what you really wanted to do was compute it in 
# pipe-stage 1 and then delay it twice and use it again in pipe-stage 
# 3 (Well, usually that's what you want). You can do that, and here's 
# how.  Instead of associating your control-bit with a list of 
# -mnemonics-, you associate it with another control-bit, by name.
# You do so thusly, using the "two-element list" form of the constructor:
#
#  e_control_bit->new
#      ([reg_write_enable => qq(=op_writes_register)]);
#
# Here, the equals-sign (=) is the que that says: "I'm just the same
# as my buddy here, only I reserve the right to live in a different
# pipe-stage."  A LATER pipe-stage, to be precise.  We currently require
# you, the human, to define the first (temporally) occurance of a control
# bit "properly," in terms of the mnemonics that set it.  You can then
# make delayed versions of that bit in later pipe-stages, and associate them
# with the earlier-version using the equals-sign.  The only rules are:
#     1) All control bits must have unique names, even delayed copies of 
#        other control bits.
#     2) Any delayed-copy created with the equals-sign (=) must be 
#        derived from a control-bit which resides in an earlier pipe-stage.
#
# **** Constructing from a hash.
#
#  Like every other e_object, you can just skip the fancy 
#  shorthand-constructors described above and justdo this:
#  do this:
#
#  e_control_bit->new 
#        ({name          => "do_zero_a",   
#          add_x_regexps => qq(MOV MOVI BR BSR JMP CALL 
#                               TRAP TRET BGEN NOT NEG ABS RDCTL),
#         });
#
#   But that would be so square.
#
# **** Pipeline accounting.
# 
# So.  How does this bit get wired to your module?  Well, let me 
# tell you how.  Since your new e_control_bit object is just
# a port (a 1-bit input port, to be precise), it will show up as 
# an input on your module.  It is also an
# e_thing_that_can_go_in_a_module, so it resides in, knows the name
# of, and can snoop on, your module (the module in which it resides).
# Later, when all the control bits are generated by the Keebler Elves
# in their hollow tree, each control bit can say which module it 
# lives in.  Modules, by name, are known to live in certain 
# pipe-stages. So the appropriate delays, etc. can be applied to
# deliver your bit, piping hot, to your pipe.  Of course, your module
# needs to have previously registered itself with the pipe-stage
# registrar, but I haven't written that part yet, so I can't tell you
# how it works.
#
# Note: Every time you new-in a control bit, it must have a unique 
#       heretofore-unknown name.  This neatly avoids the problem of 
#       two modules in two different pipe stages getting time-delayed
#       versions of the same thing.  
# 
################################################################
my %fields = (
              _order           => ["name", "add_x_regexps"],
              exclude_from_rom => 0,
  
              _x_regexp_list   => {},  # Keep as hash for uniqueness
              _is_rom          => 0,
              _alias_cbit_name => "",
              );
my %pointers = ();


&package_setup_fields_and_pointers(__PACKAGE__,
                                   \%fields,
                                   \%pointers);

################################################################################

=item I<new()>

Object constructor

=cut

sub new 
{
   my $that = shift;
   my $self = $that->SUPER::new();

   # Check for "simple string" constructor
   if (scalar (@_) == 1 && ref ($_[0]) eq "") {
      $self->_construct_from_just_a_simple_string(@_);
   } else {
      $self->set(@_);
   }

   # Check for collisions, then add to global database:
   if ((scalar(@_) == 1) && (ref($_[0]) eq __PACKAGE__))
   {
      # Copy--don't add to static list.
   } else {
      # Non-copy--do add to static list.
      &goldfish ("suspicious attempt to redefine control bit: ", $self->name())
          if $all_control_bits_by_name{$self->name()};
      $all_control_bits_by_name{$self->name()} = $self;
   }

   return $self;
}

################################################################
# parent
# 
# Override so that we can do a sanity-check: our parent must be 
# an e_pipe_module, not just a reg'lar module.
#
################################################################
################################################################################

=item I<parent()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub parent
{
   my $this = shift;
   return $this->SUPER::parent() unless @_;
   my $new_parent = shift;
   &ribbit ("too many arguments") if @_;
   &ribbit ("e_module argument required") 
       unless &is_blessed ($new_parent) && $new_parent->isa("e_module");

   &ribbit ("invalid attempt to add control-bit ",
            $this->name(), " to non-pipe-module ", 
            $new_parent->name())  
       unless $new_parent->isa("e_pipe_module");

   return $this->SUPER::parent ($new_parent);
}

################################################################################

=item I<_construct_from_just_a_simple_string()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _construct_from_just_a_simple_string
{
   my $this          = shift;
   my $simple_string = shift or &ribbit ("missing argument: string.");
   
   &ribbit ("badly-formed 'simple' name for control bit: $simple_string \n",
            "   (must start with 'do_i')\n")
       unless $simple_string =~ /^\s*do_i(\w+)\s*$/;
   
   $this->add_x_regexps ($1);
   $this->name($simple_string);
   return $this;
}

################################################################################

=item I<get_all_control_bits()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_all_control_bits
{
   my $this = shift;
   &ribbit ("access-only function") if @_;
   &ribbit ("Please call this function statically") unless ref($this) eq "";
   return values (%all_control_bits_by_name);
}

################################################################################

=item I<get_rommed_control_bit_list()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_rommed_control_bit_list
{
  my $this = shift;
  &ribbit ("access-only function") if @_;
  &ribbit ("Please call this function statically") unless ref($this) eq "";
  &ribbit(
    "Call 'allocate_rom_control_bits' before 'get_rommed_control_bit_list'\n"
  )
    if (!@_rommed_control_bits);
  return @_rommed_control_bits;
}
   
################################################################################

=item I<get_rom_contents()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_rom_contents
{
   my $this = shift;
   return @_rom_contents;
}

################################################################################

=item I<add_x_regexps()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_x_regexps
{
  my $this                = shift;
  my $pattern_list_string = shift or &ribbit ("missing list argument");

  &ribbit ("expected reference to space-separated string of mnemonic-patterns")
      unless ref ($pattern_list_string) eq "";

  # tolerate commas in pattern-list string:  by removing them!
  $pattern_list_string =~ s/\,/ /sg;
  $pattern_list_string =~ s/\n/ /sg;   # Newlines are just whitespace.
  $pattern_list_string =~ s/^\s+//sg;  # Blast leading/trailing whitespace.
  $pattern_list_string =~ s/\s+$//sg;  # Blast leading/trailing whitespace.

  foreach my $pattern (split (/\s+/, $pattern_list_string))
  { 
     # Don't mnemonic-validate equals-sign references to other named
     # control bits:
     if ($pattern !~ /^\=(.*)$/) {
        &goldfish ("pattern '$pattern' does not match any known mnemonic.")
            unless e_mnemonic->is_valid_regexp ($pattern);
     }
     
     $this->_x_regexp_list()->{$pattern}++;
  }
}

################################################################################

=item I<width()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub width
{
   my $this = shift;
   if (@_) {
      &ribbit ("can't change the width of a control bit") 
          unless $_[0] == 1;
   }
   return $this->SUPER::width();
}

################################################################################

=item I<direction()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub direction
{
   my $this = shift;
   if (@_) { 
      &ribbit ("can't change the direction of a control bit") 
          unless $_[0] eq "in";
   }
   return $this->SUPER::direction();
}

################################################################################

=item I<get_regexps()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_regexps
{
   my $this = shift;
   &ribbit ("access-only") if @_;
   return keys (%{$this->_x_regexp_list()});
}

################################################################################

=item I<get_stage_number()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_stage_number
{
   my $this = shift;
   &ribbit ("access-only") if @_;
   return $this->parent_module()->get_stage_number();
}

################################################################################

=item I<is_simple_alias()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub is_simple_alias
{
   my $this = shift;
   &ribbit ("access-only") if @_;
   my @regexp_list = $this->get_regexps();
   foreach my $regexp (@regexp_list) 
   {
      return 0 unless $regexp =~ /^\=(.*)/; 
   }
   return 1;
}

################################################################################

=item I<does_depend_on_subinstruction()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub does_depend_on_subinstruction
{
   my $this = shift;
   &ribbit ("access-only") if @_;
   return 0 if 
       $this->get_stage_number() <= 
           e_control_bit->subinstruction_origin_stage_num() + 1;

   my @regexp_list = $this->get_regexps_recursively();
   foreach my $regexp (@regexp_list) {
################################################################################

=item I<regexps()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

      next unless $regexp =~ /_(n|\d+)$/;    # Ignore non-sub regexps.
      
      # subinstruction-regexps that don't match anything don't count:
      #  (e.g. MSTEP_1 when MSTEP isn't implemented).
      #
      next if scalar (e_mnemonic->get_matching_mnemonics($regexp)) == 0;
      
      # Well, we found a subinstruction-specific regexp that actually
      # matches an implemented instruction. Depend on it:
      return 1;
   }
   # Made it through the loop without any dependencies.
   # Here's my declaration of independence:
   return 0;
}

################################################################################

=item I<make_rom_column_list()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub make_rom_column_list
{
   my $this = shift;
   &ribbit ("too many arguments") if @_;
   
   # Start off with a list of zeroes.
   my @result = ();
   for my $i (0..127) {
      push (@result, 0);
   }

   my @mnem_list = 
       e_mnemonic->get_matching_mnemonics($this->get_regexps_recursively());

   # This utterly relies on the fact that no mnemonic will have a 
   # downcount value greater than one.  So stated.
   foreach my $mnem (@mnem_list) 
   {
      my $bitstring  = $mnem->bit_string();
      my $mnem_table = $mnem->table()->name(); 

      if      ($mnem_table =~ /major/i) {
         $bitstring = "0" .$bitstring;        # 1st half, 6-bit opcode.
      } elsif ($mnem_table =~ /U/i) {
         $bitstring = "10".$bitstring. "x0";  # 3rd quarter, 3-bit op, lsb=0 
      } elsif ($mnem_table =~ /V/i) {
         $bitstring = "10".$bitstring."xx1";  # 3rd quarter, 2-bit op, lsb=1
      } elsif ($mnem_table =~ /W/i) {
         $bitstring = "11".$bitstring;        # quarter 4, 5-bit opcode.
      }
      &fill_in_ones (\@result, [&convert_to_ordinals($bitstring)]);
   }
   return @result;
}   

################################################################################

=item I<fill_in_ones()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub fill_in_ones 
{
   my $destination_list = shift;
   my $source_list      = shift;
   foreach my $one_index (@{$source_list}) 
   {
      $destination_list->[$one_index] = 1;
   }
}

################################################################################

=item I<convert_to_ordinals()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub convert_to_ordinals
{
   my $bitstring = shift;
   my @result = ();
   &ribbit ("bad bitstring: '$bitstring'") 
       if  $bitstring =~ /[^10x]/;

   if ($bitstring !~ /x/) {
      return (&bitstring_to_num($bitstring));
   } else {
      my $one_case = $bitstring;
      my $zero_case = $bitstring;
      $one_case  =~ s/x/1/;      # just the first one, please.
      $zero_case =~ s/x/0/;      # just the first one, please.
      return (&convert_to_ordinals($zero_case),
              &convert_to_ordinals($one_case) ,);
   }
}

################################################################################

=item I<bitstring_to_num()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub bitstring_to_num
{
   my $bitstring = shift;
   my @bits = split(//, $bitstring);
   my $result = 0;
   foreach my $bit (@bits)
   {
      $result *= 2;
      $result++ if $bit eq "1";
   }
   return $result;
}
   
################################################################
# get_regexps_recursively
#
# Control-bits can refer to mnemonics, or they can refer to other
# control-bits which, themselves, refer to mnemonics.  
# 
# Sometimes, you want to know -all- the mnemonics from whence a control-bit
# comes, even the ones that are subsumed within an "alias" control-bit 
# sub-part.  If so, this is your function.
#
# You remember, of course, that our "alias" references to other control
# bits are marked by a leading equals-sign (=).
#
################################################################
################################################################################

=item I<get_regexps_recursively()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_regexps_recursively
{
   my $this = shift;
   &ribbit ("access-only") if @_;
   # Keep track of all sub-parts in a hash, to avoid duplicates:
   my %regexp_hash = ();   
   foreach my $regexp (keys (%{$this->_x_regexp_list()}))
   {
      if ($regexp =~ /^\=(.*)$/) 
      {
         my $sub_cbit = $all_control_bits_by_name{$1};
         my @sub_regexp_list = $sub_cbit->get_regexps_recursively();
         foreach my $sub_regexp (@sub_regexp_list) {
            $regexp_hash {$sub_regexp}++ ;
         }
      } else {
         $regexp_hash {$regexp}++ ;
      }
   }
   return keys (%regexp_hash);
}

################################################################
# get_regexps_by_name
#
# If you know the name of a control-bit, this will return a list
# of all its regexps.  You must call this statically.
#
################################################################
################################################################################

=item I<get_regexps_by_name()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_regexps_by_name
{
   my $this = shift;
   &ribbit ("Please call statically") unless ref ($this) eq "";

   my @result = ();
   foreach my $bit_name (@_)
   {
      my $control_bit_object = $all_control_bits_by_name{$bit_name}
        or &ribbit ("'$bit_name' is not a known control bit") ;
      push (@result, $control_bit_object->get_regexps());
   }            
   return @result;
}

# Make and store all data relating to control bit ROM.
# - control bits themselves
# - ROM contents array
# - MIF file
# - DAT file
# 
# Returns the name of the mif file which contains Quartus-compilation
# ROM contents, in case the caller cares.  The caller cares, because
# they need to bind their e_ram to that file.
#
# Important Notice of Laziness: 
# Writing MIF and DAT files properly lies under the bailiwick of e_ram
# and e_rom.  We do it here because we're lazy and don't want to change
# working code too much.  If you're reading this code with an eye to create
# contents for your own e_ram or e_rom object, please consider moving the
# code out of here, and sticking it in e_lpm_instance for all to enjoy.
#
################################################################################

=item I<initialize_rom_contents()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub initialize_rom_contents
{
   my $this = shift;
   my ($Opt, $project) = (@_);
   
   $this->allocate_rom_control_bits(@_);
   $this->create_rom_contents(@_);
   
   # Synthesize a few more parameters which will be needed in order to
   # make happy fun ROM files.
   my $rom_width = scalar(e_control_bit->get_rommed_control_bit_list());
   my $rom_depth = scalar(e_control_bit->get_rom_contents());

   my $file_base = $Opt->{name} . '_instruction_decoder_rom';
   my $mif_name = $file_base . ".mif";
   my $dat_name = $file_base . ".dat";
   my $hex_name = $file_base . ".hex";
   
   $this->create_rom_files(
     {
       mif_name => $mif_name,
       dat_name => $dat_name,
       hex_name => $hex_name,
       rom_width => $rom_width,
       rom_depth => $rom_depth,
       $Opt,},
     $project,
   );
  
   return $file_base;
}

my @bitstring_report = ();

# Setting this state-flag to 1 at the beginning of time
# disables the "rom-allocation" feature, which is what we want for now.
#
################################################################################

=item I<allocate_rom_control_bits()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub allocate_rom_control_bits
{
   my $this = shift;
   my ($Opt, $project) = (@_);
   
   &ribbit ("Please call statically") unless ref ($this) eq "";
   my %score_table = ();
   foreach my $cbit (e_control_bit->get_all_control_bits()) {
      # A human may have given us a hint that this bit might be better-off
      # outside the ROM-table
      #
      next if $cbit->exclude_from_rom();
      my %matched_mnemonics = (); # Use hash to avoid double-counting of mnems
      foreach my $x_regexp ($cbit->get_regexps_recursively()) 
      {
         my @match_list = e_mnemonic->get_matching_mnemonics($x_regexp);
         foreach my $mnemonic (@match_list) {
            $matched_mnemonics{$mnemonic->name()}++;
         }
      }
      my $num_matches = scalar(keys(%matched_mnemonics));
      my $score = $num_matches;

      # Put some bias on rom-bits from earlier stages (penalize
      #  bits in later stages).  We like bits in earlier stages because
      # then we don't have to make as many delay-LEs to get them to 
      # where they're going.  Bits which require a lot of delayin' 
      # can, in general, use those delay-LEs to do some decode-logic
      # at zero penalty, minimizing the advantage of ROMming them.
      #
      my $stage_num = $cbit->get_stage_number();
      $score += e_pipe_module->get_max_delay() - $stage_num;

      # Can't rommify control-bits used before stage #2, because,
      # using a ROM, it takes two stages to come up with them.
      $score = 0 if $stage_num < 1;
      $score = 0 if $cbit->is_simple_alias();
      if ($score && $cbit->does_depend_on_subinstruction()) {
         if ($Opt->{verbose}) {
            print STDERR "disqualifying ($score)", $cbit->name(), "\n";
         }
         $score = 0;
      }
      $score_table{$cbit->name()} = $score;
   }

   my @sorted_list 
       = reverse( sort {$score_table{$a} <=> $score_table{$b}} 
         keys(%all_control_bits_by_name));

   foreach my $i (0..$Opt->{num_rom_control_bits}-1) 
   { 
      my $bit_name = $sorted_list[$i];
      &ribbit ("ERROR: not enough qualified control-bits for decoder-ROM.\n")
          if $score_table{$bit_name} <= 0;
      print STDERR "ROMMED: $bit_name      ($score_table{$bit_name})\n"
          if ($Opt->{verbose});
      my $rommed_bit = $all_control_bits_by_name{$bit_name};
      $rommed_bit->_is_rom(1);

      # Push onto global ordered-list.
      push (@_rommed_control_bits, $rommed_bit);
   }

}

################################################################################

=item I<create_rom_contents()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub create_rom_contents
{
   my $this = shift;
   my ($Opt, $project) = (@_);

   my @rommed_control_bits = $this->get_rommed_control_bit_list();

   # From the just-computed control bits, compute ROM contents.
   my @result = ();
   foreach my $control_bit (@rommed_control_bits)
   {
      my @column = $control_bit->make_rom_column_list();
      my $i = 0;
      foreach my $bit (@column)  {
         $_rom_contents[$i++] .= $bit;
      }
   }
   if ($Opt->{verbose}) 
   {
      print STDERR "Decoder-Rom might look something like this:\n";
      foreach my $bitstring (@_rom_contents)
      {
         print STDERR "  $bitstring\n";
      }
      print STDERR "\n";
   }
   
}

################################################################################

=item I<create_rom_files()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub create_rom_files
{
   my $this = shift;
   my ($Opt, $project) = (@_);
   
   # Write the MIF file for the not-yet-created ROM.
   ################
   # Mif-file generation.
   #
   
   &ribbit ("decoder-roms limited to 32-bits wide") 
       if $Opt->{rom_width} > 32;     # Enforce arbitrary limit.
   open (MIFFILE, ">$Opt->{mif_name}") 
       or &ribbit ("couldn't open $Opt->{mif_name}: $!");
   print MIFFILE "WIDTH=$Opt->{rom_width};\n";
   print MIFFILE "DEPTH=$Opt->{rom_depth};\n";
   print MIFFILE "ADDRESS_RADIX=HEX;\n";
   print MIFFILE "DATA_RADIX=HEX;\n";
   print MIFFILE "CONTENT BEGIN\n";
   my $addr = 0;
   
   my @rom_contents = $this->get_rom_contents();
   foreach my $bitstring (@rom_contents)
   {
      print MIFFILE sprintf ("  %08X : %08X;\n", 
                             $addr++, 
                             &bitstring_to_num($bitstring));
   }
   print MIFFILE "END;\n";
   close (MIFFILE);

   ################
   # Make a DAT file for simulation.
   # 
   # (fcu_convert reads the MIF file in, and writes a
   # corresponding DAT file out.)
   my $simdir = $project->simulation_directory();
     &fcu_convert ({"0"  => $Opt->{mif_name},
                   "1"  => $simdir . "/" . $Opt->{dat_name},
                   oformat => "dat",
                   width   => $Opt->{rom_width},
                  });

   ################
   # Jumping Jesus on a Lizard!!! We need 'nother memory simulation file!?!?!
   # Yup, ya'll we dun be usin' da' Altera sim'lation models files for memories.
   # They needz HEX files, or sumthin'.
     &fcu_convert ({"0"  => $Opt->{mif_name},
                   "1"  => $simdir . "/" . $Opt->{hex_name},
                   oformat => "hex",
                   width   => $Opt->{rom_width},
                  });
}

################################################################
# build_rom_module
#
# Returns you a -list- of e_module-objects, which are prototypes 
# of all the decoder-rom objects.  Each decoder-rom has its own
# prototype (e_module object).  You, yourself, have to 
# instantiate these things in some higher-level module.
#
################################################################
# sub build_rom_module
# {
#    my $this = shift;
#    my ($Opt, $project) = (@_);
# 
#    &ribbit ("Please call statically") unless ref ($this) eq "";
# 
#    &goldfish ("Strange. build_rom_module called non-statically")
#        unless ref ($this) eq "";
# 
#    # This function decides which bits should be rommed--and 
#    # returns an ordered list of the rommed bits:
#    #
#    e_control_bit->allocate_rom_control_bits($Opt, $project);
#    my @rommed_list = $this->get_rommed_control_bit_list();
# 
#    my @result = ();
#    my @contents = ();
#    foreach my $control_bit (@rommed_list)
#    {
#       my @column = $control_bit->make_rom_column_list();
#       my $i = 0;
#       foreach my $bit (@column)  {
#          $contents[$i++] .= $bit;
#       }
#    }
#    if ($Opt->{verbose}) 
#    {
#       print STDERR "Decoder-Rom might look something like this:\n";
#       foreach my $bitstring (@contents)
#       {
#          print STDERR "  $bitstring\n";
#       }
#       print STDERR "\n";
#    }
#    push (@result, &make_decoder_rom_module 
#                        ({Opt       => $Opt,
#                          project   => $project, 
#                          rom_name  => "instruction_decoder_rom",
#                          rom_width => scalar (@rommed_list),
#                          contents  => \@contents,
#                        })
#           );
#    return @result;
# }

################################################################
# make_decoder_rom_module
#
# Each ROM we make gets its own dat/mif file, and therefore 
# requires its own private prototype.
#
# This function builds the prototype-module.
#
# You pass this function a SINGLE $arg-hash with the following 
# known (and required) keys:
#   1) Opt       => The usual CPU-wide "$Opt" hash.
#   2) project   => The usual $project.
#   3) rom_name  => The base-name of this particular module (prototype name).
#   3) rom_width => The number of output-bits from the ROM.
#   4) contents  => A -list- of binary strings that specify the contents.
#
# This thing will make the necessary MIF/DAT files
#
# Since we need our ROMs to have a "readclken" signal for 
# pipeline-compliance, we are forced into the nonintuitive position
# of using "lpm_dpram" objects to implement the ROM.  This doesn't 
# really hurt anything, it's just...strange.
#
# The astute reader will note that this looks just like the function
# "make_dpram_module" in the committed pipeline.  Careful thinkin' might
# result in a Grand Unification.  Fly at it, Einstein.
#
################################################################
# sub make_decoder_rom_module
# {
#    my $arg     = shift           or &ribbit ("No args!");
#    my $Opt     = $arg->{Opt}     or &ribbit ("missing Opt-hash");
#    my $project = $arg->{project} or &ribbit ("missing project");
# 
#    my $module_name      = $Opt->{name}."_".$arg->{rom_name};
#    $arg->{depth}        = scalar (@{$arg->{contents}});
#    $arg->{address_bits} = &Bits_To_Encode($arg->{depth} - 1);
# 
#    ################
#    # Mif-file generation.
#    #
#    $arg->{mif_name}    = $module_name . ".mif";
#    &ribbit ("decoder-roms limited to 32-bits wide") 
#        if $arg->{rom_width} > 32;     # Enforce arbitrary limit.
#    open (MIFFILE, ">$arg->{mif_name}") 
#        or &ribbit ("couldn't open $arg->{mif_name}: $!");
#    print MIFFILE "WIDTH=$arg->{rom_width};\n";
#    print MIFFILE "DEPTH=$arg->{depth};\n";
#    print MIFFILE "ADDRESS_RADIX=HEX;\n";
#    print MIFFILE "DATA_RADIX=HEX;\n";
#    print MIFFILE "CONTENT BEGIN\n";
#    my $addr = 0;
#    foreach my $bitstring (@{$arg->{contents}}) {
#       print MIFFILE sprintf ("  %08X : %08X;\n", 
#                              $addr++, 
#                              &bitstring_to_num($bitstring));
#    }
#    print MIFFILE "END;\n";
#    close (MIFFILE);
# 
#    ################
#    # Dat-file conversion.
#    # 
#    my $simdir = $project->simulation_directory();
#      $arg->{dat_name} = $module_name . ".dat";
#      &fcu_convert ({"0"  => $arg->{mif_name},
#                    "1"  => $simdir . "/" . $arg->{dat_name},
#                    oformat => "dat",
#                    width   => $arg->{rom_width},
#                   });
#    
#    # Simulation model partitioned-off into separate module:
#    #
#    # my $sim_model = &make_decoder_rom_sim_model ($arg);
# 
# 
#    ################
#    # Finally--start constructing "wrapper" module:
#    #
#    my $module    = e_module->new ({name => $module_name});
#    # $module->_explicitly_empty_module (1);
#    # $module->do_black_box (1);
#    $project->add_module($module);
#    # my $marker = e_default_module_marker->new ($module);
# 
#    if ($Opt->{use_altsyncram})
#    {
#        # We need to instantiate different macrofunctions depending on whether
#        # the target device has the new TriMatrix memory.
#        #
#        # Actually . . . we may not NEED to.
#        #
#        # If we were to confirm that using altsyncram doesn't affect operation
#        # in other families, we might eliminate this check, using altsyncram
#        # always.  But for now, the safest thing is to leave the code for
#        # 20KE, etc as is, and add code for Stratix and later families.
#        #
# =head       
#        e_port->adds 
#            ([q_a         => $arg->{rom_width},   "out"],
#             [clock0      => 1,                   "in" ],
#             [clocken0    => 1,                   "in" ],
#             [address_a   => $arg->{address_bits},"in" ],  );
#             
#        e_parameter->adds
#            ([qw(lpm_type               altsyncram            STRING  ) ],
#             [qw(operation_mode         ROM                   STRING  ) ],
#             [  "width_a",              $arg->{rom_width},   "INTEGER"  ],
#             [  "widthad_a",            $arg->{address_bits},"INTEGER"  ],
#             [  "num_words_a",          $arg->{depth},       "INTEGER"  ],
#             [qw(outdata_reg_a          UNREGISTERED          STRING  ) ],
#             [  "init_file",            $arg->{mif_name},    "STRING"   ],  );
# 
# 
#        e_instance->add
#            ({module => $sim_model->name(),
#              tag    => "simulation",
#              parameter_map => {operation_mode        => "operation_mode",
#                                width_a               => "width_a",
#                                widthad_a             => "widthad_a",
#                                num_words_a           => "num_words_a",
#                                outdata_reg_a         => "outdata_reg_a",
#                                init_file             => "init_file",
#                             },
#           });
# =cut
#    } else {
#        # This thing -is- a lpm_ram_dp, so the poor user is required to 
#        # tie-off the "stupid" ports.  Sorry. Fortunately, not that many
#        # people ever instantiate this thing.
#        #
#       e_ram->add({
#          name => $module_name . "_rom",
#          Read_Latency => 1,
#          implement_as_esb => 1,
#          mif_file => $arg->{mif_name},
#       });
# =head       
#        e_port->adds 
#            ([wren      => 1,                   "in" ],
#             [wrclock   => 1,                   "in" ],
#             [rdclken   => 1,                   "in" ],
#             [rdclock   => 1,                   "in" ],
#             [rdaddress => $arg->{address_bits},"in" ],
#             [wraddress => $arg->{address_bits},"in" ],
#             [data      => $arg->{rom_width},   "in" ],
#             [q         => $arg->{rom_width},   "out"],  );
#             
#        e_parameter->adds
#            ([qw(lpm_type               lpm_ram_dp            STRING  ) ],
#             [  "lpm_width",            $arg->{rom_width},   "INTEGER"  ],
#             [  "lpm_widthad",          $arg->{address_bits},"INTEGER"  ],
#             [  "lpm_file",             $arg->{mif_name},    "STRING"   ],
#             [qw(lpm_indata             REGISTERED            STRING  ) ],
#             [qw(lpm_outdata            UNREGISTERED          STRING  ) ],
#             [qw(lpm_wraddress_control  REGISTERED            STRING  ) ],
#             [qw(lpm_rdaddress_control  REGISTERED            STRING  ) ],
#             [qw(lpm_hint               USE_EAB=ON            STRING  ) ],  );
#     
#     
#        e_instance->add
#            ({module => $sim_model->name(),
#              tag    => "simulation",
#              parameter_map => {lpm_width             => "lpm_width",
#                                lpm_widthad           => "lpm_widthad",
#                                lpm_file              => "lpm_file",
#                                lpm_indata            => "lpm_indata",
#                                lpm_outdata           => "lpm_outdata",
#                                lpm_wraddress_control => "lpm_wraddress_control",
#                                lpm_rdaddress_control => "lpm_rdaddress_control",
#                                lpm_hint              => "lpm_hint",
#                             },
#           });          
# =cut
#    }
#    
#    return $module;
# }

# ################################################################
# # make_decoder_rom_sim_model
# #
# # Evil-twin companion function of "make_decoder_rom_module", above.
# # Read the novella-comment there.  Einstein.
# #
# ################################################################
################################################################################

=item I<make_decoder_rom_sim_model()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub make_decoder_rom_sim_model
{
   my $arg     = shift           or &ribbit ("No args!");
   my $Opt     = $arg->{Opt}     or &ribbit ("missing Opt-hash");
   my $project = $arg->{project} or &ribbit ("missing project");

   my $module_name = $Opt->{name}."_".$arg->{rom_name}."_sim_model";

   my $module = e_module->new ({name => $module_name});
   $module->do_black_box (1);
   $module->_hdl_generated (0);   # Make 1 if you really want to use lib cell.
   $module->_explicitly_empty_module (1);
   $project->add_module($module);
   my $marker = e_default_module_marker->new ($module);

   if ($Opt->{use_altsyncram})
   {
       # Use ALTSYNCRAM
       e_port->adds 
           ([q_a         => $arg->{rom_width},   "out"],
            [clock0      => 1,                   "in" ],
            [clocken0    => 1,                   "in" ],
            [address_a   => $arg->{address_bits},"in" ],  );
            
       e_parameter->adds
           ([qw(lpm_type               altsyncram            STRING  ) ],
            [qw(operation_mode         ROM                   STRING  ) ],
            [  "width_a",              $arg->{rom_width},   "INTEGER"  ],
            [  "widthad_a",            $arg->{address_bits},"INTEGER"  ],
            [  "num_words_a",          $arg->{depth},       "INTEGER"  ],
            [qw(outdata_reg_a          UNREGISTERED          STRING  ) ],
            [  "init_file",            $arg->{mif_name},    "STRING"   ],  );

       e_signal->add ({name         => "mem_array",
                       width        => $arg->{rom_width},
                       depth        => $arg->{depth},
                       never_export => 1,
                       tag          => "simulation",
                    });

       e_signal->add ({name         => "p1_q",
                       width        => $arg->{rom_width},
                       never_export => 1,
                       tag          => "simulation",
                    });

       e_assign->add (["p1_q", "mem_array[address_a]"])->tag("simulation");

       e_register->add 
           ({out       => "q_a",   
             in        => "p1_q",  
             clock     => "clock0",
             enable    => "clocken0",
             tag       => "simulation",
             async_set => "1'b1",
          });

       e_initial_block->add({
          tag      => "simulation",
          # Bogus compensation for basic misfeature
          #  of "e_initial_block" object:
          clock        => "clock0",   
          contents => [
                       e_readmem->new({
                          tag          => "simulation",
                          file         => $arg->{dat_name},
                          mem_variable => "mem_array",
                          hex_output   => 1,
                       }),
                       ],
       });
   } else {
       # Use LPM_RAM_DP
       e_port->adds 
           ([wren      => 1,                   "in" ],
            [wrclock   => 1,                   "in" ],
            [rdclken   => 1,                   "in" ],
            [rdclock   => 1,                   "in" ],
            [rdaddress => $arg->{address_bits},"in" ],
            [wraddress => $arg->{address_bits},"in" ],
            [data      => $arg->{rom_width},   "in" ],
            [q         => $arg->{rom_width},   "out"],  );
            
       e_parameter->adds
           ([qw(lpm_type               lpm_ram_dp            STRING  ) ],
            [  "lpm_width",            $arg->{rom_width},   "INTEGER"  ],
            [  "lpm_widthad",          $arg->{address_bits},"INTEGER"  ],
            [  "lpm_file",             $arg->{mif_name},    "STRING"   ],
            [qw(lpm_indata             REGISTERED            STRING  ) ],
            [qw(lpm_outdata            REGISTERED            STRING  ) ],
            [qw(lpm_wraddress_control  REGISTERED            STRING  ) ],
            [qw(lpm_rdaddress_control  UNREGISTERED          STRING  ) ],
            [qw(lpm_hint               USE_EAB=ON            STRING  ) ],  );

       e_signal->add ({name         => "mem_array",
                       width        => $arg->{rom_width},
                       depth        => $arg->{depth},
                       never_export => 1,
                       tag          => "simulation",
                    });

       e_signal->add ({name         => "p1_q",
                       width        => $arg->{rom_width},
                       never_export => 1,
                       tag          => "simulation",
                    });

       e_assign->add (["p1_q", "mem_array[rdaddress]"])->tag("simulation");

       e_register->add 
           ({out       => "q",   
             in        => "p1_q",  
             clock     => "rdclock",
             enable    => "rdclken",
             tag       => "simulation",
             async_set => "1'b1",
          });

       e_initial_block->add({
          tag      => "simulation",
          # Bogus compensation for basic misfeature
          #  of "e_initial_block" object:
          clock        => "rdclock",   
          contents => [
                       e_readmem->new({
                          tag          => "simulation",
                          file         => $arg->{dat_name},
                          mem_variable => "mem_array",
                          hex_output   => 1,
                       }),
                       ],
       });
   }
   return $module;
}

my %cbit_count_by_stage = ();

################################################################################

=item I<_implement_alias_logic()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _implement_alias_logic
{
   my $this = shift;
   my $source_bit_name = shift or &ribbit ("no source-bit name given");
   my $source_bit = $all_control_bits_by_name{$source_bit_name};

   &ribbit ("Can't implement control-bit ", 
            $this->name(), 
            " as an alias of unknown source-bit '$source_bit_name'.")
       unless $source_bit;
   
   my $source_stage_num = $source_bit->parent_module()->get_stage_number();
   my $stage_num        = $this      ->parent_module()->get_stage_number();

   &ribbit ("Sorry, you may only create aliases of control-bits 
             defined in previous (earlier) stages.  
             Can't make '",$this->name(),"' ($stage_num) from
             $source_bit_name ($source_stage_num)"                 ) 
       if $source_stage_num > $stage_num;

   my $alias_name = sprintf ("%s_delayed_for_%s", 
                             $source_bit_name, $this->name);

   
   if ($stage_num == $source_stage_num) {
      $source_bit_name = "p_$source_bit_name";
   } else {
      $stage_num -= 1 unless $stage_num == 0 ;
   }
   # We return a list.  The first thing is the name of the or-term 
   # we just created.  All other things on the list are 
   # logic-items that implement whatever delay (or assignment) is 
   # required.
   #
   my @result = ($alias_name);
   
   # Create assignments that "tie off" the beginning and end of the delay
   # chain with regularized names.
   push (@result, 
         e_assign->new(["$alias_name\_$source_stage_num", $source_bit_name]));
   push (@result, 
         e_assign->new([$alias_name, "$alias_name\_$stage_num"]));

   foreach my $i ($source_stage_num+1 .. $stage_num) 
   {
      my $prev = $i-1;
      my $local_clk_en = e_pipe_module->get_stage_clk_en_signal($prev);
      push (@result, e_register->new ({out    => "$alias_name\_$i",
                                       in     => "$alias_name\_$prev",
                                       enable => $local_clk_en,
                                    }));
   }
   return @result;
}

################################################################################

=item I<implement_logic()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub implement_logic 
{
   my $this = shift;
   my $instruction_word_signal_name = shift 
       or &ribbit ("Expected instruction-word signal name argument");
   ref ($instruction_word_signal_name) eq "" or &ribbit ("expected string.");
   my $subinstruction_signal_name = shift;

   &ribbit ("too many arguments") if @_;


   my $stage_num = $this->get_stage_number();

   # In general, we decode the instruction from the previous stage,
   # then run the result into one delay.  The one exception 
   # is stage 0, where we don't get the luxury of an output-delay.
   # Alas.
   my $logic_stage = $stage_num - 1;
   $logic_stage = 0 if $stage_num == 0;

   # Rom-bits pop out of the oven in stage 1, the earliest opportunity
   # to produce "p1_<bit_name>".
   $logic_stage = 1 if ($this->_is_rom());

   my $suffix = "_" . ($logic_stage);

   $instruction_word_signal_name .= $suffix;
   # Note that the subinstruction-part doesn't actually come into 
   # existence until some point in the middle of the pipeline (early,
   # but not at the beginning).  This is a "special event," which 
   # e_control_bits in-general must have been previously notified about.
   # 
   
   $subinstruction_signal_name .= $suffix;
   $subinstruction_signal_name = ""
       if $stage_num <= e_control_bit->subinstruction_origin_stage_num() + 1;

   # Push all the objects that implement me onto this fine list.
   # We know we're going to have (at least) one port-thing:
   #
   my $out_port = e_port->new ([$this->name()=> 1, "out"]);
   my @result = ($out_port);

   my @or_terms      = ("1'b0");
   my @comment_terms = ("Control-bit ".$this->name().", set by: ");
   my @bitstrings = ();

   my %matched_mnemonics = ();   # Just for scorekeeping/reporting.
   if (!$this->_is_rom()) {
      foreach my $x_regexp ($this->get_regexps()) 
      {
         # Regexps can either be familiar mnemonic-things, or they
         # can be references to other control-bits by-name (preceded
         # by an equals-sign).  Examples:
         #
         #   =do_iPFX        -- reference to other control bit (equals-sign).
         #   RxC_x           -- mnemonic-match expression.
         #                
         # Handle each case separately:
         #
         if ($x_regexp =~ /^\=(.*)$/)
         {
            my $original_control_bit = $1;
            # Generate logic to delay and or-in preexisting control bit.
            # 
            my ($delayed_alias_name, @delay_logic) = 
                $this->_implement_alias_logic($original_control_bit);
            push (@comment_terms, $delayed_alias_name);
            push (@or_terms, $delayed_alias_name);
            push (@result, @delay_logic);
            
         } else {
            
            my @match_list = e_mnemonic->get_matching_mnemonics($x_regexp);
            foreach my $mnemonic (@match_list) {
               $matched_mnemonics{$mnemonic}++;
               push (@comment_terms, $mnemonic->name());
               push (@or_terms, $mnemonic->make_match_expression
                                 ($instruction_word_signal_name,
                                  $subinstruction_signal_name   ));
               push (@bitstrings, e_mnemonic->get_full_bitstring
                     ($mnemonic->name(), "yes, reduction string, please."));
               
            }
         }
      }
   }

   # Handy reporting:
   my $N_bits = scalar(keys(%all_control_bits_by_name));
   my $N_mnem = e_mnemonic->count_all_mnemonics();
   my $num_matches = scalar(keys(%matched_mnemonics));
   push (@comment_terms, "\n Just one of $N_bits total control bits.\n");
   push (@comment_terms, 
         "\n Matches $num_matches of $N_mnem mnemonics, stage=$stage_num\n");
   my $logic_sig = e_signal->new (["p_".$this->name(), 1]);
   if (!$this->_is_rom()) {
      push (@result, e_assign->new ({lhs     => $logic_sig,
                                     rhs     => join (" ||\n", @or_terms     ),
                                  })); 
   }

   push (@bitstring_report, 
         $this->name(), " (\n", join (",\n", @bitstrings), "\n)\n");

   foreach my $s ($logic_stage .. $stage_num)
   {
      my $s1 = $s + 1;
      my $in_sig  = sprintf("p%d_%s", $s, $this->name());
         $in_sig  = sprintf("p_%s",       $this->name()) if $s == $logic_stage;
      my $out_sig = sprintf("p%d_%s", $s1,$this->name());
         $out_sig = $this->name()                        if $s == $stage_num;
      
      if ($s == $stage_num) {
         push (@result, e_assign->new 
                           ({lhs     => $out_sig,
                             rhs     => $in_sig,
                             comment => join (", ",    @comment_terms),
                          })  
               );
      } else {
         my $stage_enable = 
             e_pipe_module->get_stage_clk_en_signal($s);
         push (@result, e_pipe_register->new 
               ({out     => $out_sig,
                 in      => $in_sig, 
                 enable  => $stage_enable,
                })  
               );
      }    
   }

   # Keep score of how many (useful) control-bits are in each 
   # stage.
   #
   $cbit_count_by_stage{$stage_num}++ if $num_matches;
   return @result;
}

################################################################################

=item I<print_logic_report_file()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub print_logic_report_file
{
   my $this = shift;
   my $fname  = shift or &ribbit ("no filename");

   print "Control-bit counts by stage:\n  ";
   print join ("\n  ", 
               map {"$_ : $cbit_count_by_stage{$_}"} 
               keys(%cbit_count_by_stage));
   print "\n";
   
   open (EXPRFILE, ">$fname") or &ribbit("couldn't open $fname ($!)");
   print EXPRFILE join ("", @bitstring_report);

   print EXPRFILE "\n-- List of all mnemonics:\n  ";
   foreach my $mnem (e_mnemonic->get_all_counted_mnemonics()) {
      print EXPRFILE "  ", $mnem->name(), "\n";
   }
   close EXPRFILE;
}

################################################################
# subinstruction_origin_stage_num
#
# control bits in-general need to know when the subinstruction field
# comes into existence, so they can properly decode themselves.
#
################################################################
my $_subinstruction_origin_stage_num = 0;
################################################################################

=item I<subinstruction_origin_stage_num()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub subinstruction_origin_stage_num 
{
   my $this = shift;
   ref ($this) eq "" or &ribbit ("Please call statically");
   return $_subinstruction_origin_stage_num unless @_;
   return ($_subinstruction_origin_stage_num = shift);
}



"How are you gentlemen.";










=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_port

=begin html

<A HREF="e_port.html">e_port</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
