=head1 NAME

e_avalon_adapter_interface 


=head1 SYNOPSIS

This package contains subroutines needed by both e_avalon_adapter_master and
e_avalon_adapter_slave.  These routines could equavalently be placed in 
e_avalon_slave, but since the routines are very adapter-centric, it makes
more sense to put them in their own package and have the adapter interface
objects do multiple inheritance.

=head1 METHODS

=over 4

=cut

package e_avalon_adapter_interface;

use e_thing_that_can_go_in_a_module;
use e_module;
use e_signal;
use e_project;
@ISA = ("e_thing_that_can_go_in_a_module");
use europa_utils;
use strict;

my %fields = (
  # Logic may be associated with this interface.  However,
  # deriving this interface class from e_module seems troublesome.
  # Instead, throw any logic into a module owned within this interface.
  module => undef,
);

my %pointers = (
);

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

=item I<non_adapter_types()>

Here's a list of types that adapters don't need
to handle.
These ones are special because they are not simply
outputs from a (slave, master) and input to the (master, slave).
 

=cut

=item I<add_to_type_map()>

Call this function to build up the type map "on-the-fly",
that is, as the owning object requests ports and signals by type.

=cut

sub add_to_type_map($$$)
{
  my $this = shift;
  my $type = shift;
  my $signal = shift;

  $this->parent_module()->get_and_set_once_by_name({
    thing => 'signal',
    name => $signal,
    width => $this->get_port_width_by_type($type),
  });
  $this->type_map()->{$signal} = $type;
}


=item I<get_port_by_type()>

Make an assignment to the signal type which this interface provides
to its owner.

=cut

sub get_port_by_type($$)
{
  my $this = shift;
  my $type = shift;

  my $signal = $this->get_signal_by_type($type);
  $this->add_to_type_map($type, $signal);
  
  return $signal;
}

=item I<get_port_width_by_type()>

Return widths of interface port signals.

=cut

sub get_port_width_by_type($$)
{
  my $this = shift;
  my $type = shift;
  
  my $width = 1;
  
  if ($type =~ /data(_n)?$/)
  {
    $width = $this->SBI_section()->{Data_Width};
  }
  elsif ($type =~ /byteenable(_n)?/)
  {
    $width =
      round_up_to_next_computer_acceptable_bit_width(
        $this->SBI_section()->{Data_Width}
      ) / 8;
  }
  elsif ($type =~ /^address(_n)?$/)
  {
    $width = $this->SBI_section()->{Address_Width};
  }
  elsif ($type =~ /^nativeaddress$/)
  {
    $width = $this->SBI_section()->{Address_Width};
    # Adjust it so it's a native address.
    $width -= log2(
      round_up_to_next_computer_acceptable_bit_width(
        $this->SBI_section()->{Data_Width}
      ) / 8);
  }
  elsif ($type =~ /^byteaddress(_n)?$/)
  {
    if ($this->section_name() eq 'SLAVE')
    {
      $width = $this->SBI_section()->{Address_Width};
      # Adjust it so it's a byte address.
      $width += log2(
        round_up_to_next_computer_acceptable_bit_width(
          $this->SBI_section()->{Data_Width}
        ) / 8);
    }
    else
    {
      # Master address widths are already expressed in byte-aligned terms.
      $width = $this->SBI_section()->{Address_Width};
    }
  }
  elsif ($type =~ /^burstcount(_n)?$/)
  {
    $width = &Bits_To_Encode($this->SBI_section()->{Maximum_Burst_Size});

    # It can happen that the adapter slave/master doesn't have a
    # max burst size assignment (for example, if the original slave
    # in front of which the adapter was inserted had no such assignment.
    # In such cases,  max burst size of "1" is appropriate.
    $width = 1 if !$width;
  }
  
  return $width;
}

=item I<get_signal_by_type()>

Return the name of the signal which represents a particular
Avalon type to the export side of the interface.

=cut

sub get_signal_by_type($$)
{
  my $this = shift;
  my $type = shift;

  # We could vary the signal name based on its characteristics
  # within the interface, e.g. pipelined, fifo'ed, etc.  For now,
  # use a simpler rule.
  
  my $signal = $this->name() . "_$type";

  $this->add_to_type_map($type, $signal);

  return $signal;
}
 
sub max_pending_reads
{
  my $this = shift;

  my $mpr = $this->SBI_section()->{Maximum_Pending_Read_Transactions};
  return $mpr;
}


1;
