--
--Copyright (c) 1996 by Altera Corporation.  All rights reserved.  This text
--contains proprietary, confidential information of Altera Corporation, and
--may be used, copied, and/or disclosed only pursuant to the terms of a
--valid software license agreement with Altera Corporation.  This copyright
--notice must be retained as part of this text at all times.
--

LIBRARY ieee;
USE ieee.std_logic_1164.all;

ENTITY clklock IS
-- pragma translate_off
     GENERIC (	clockboost: INTEGER := 1;
                input_frequency : REAL := 50.00;
		tpd : TIME := 0 ns );
-- pragma translate_on
     PORT (
             outclk : OUT STD_LOGIC;  
             inclk  : IN STD_LOGIC);
END clklock;

ARCHITECTURE behavior OF clklock IS

-- pragma translate_off
  SIGNAL clklock_half_period : TIME;
  SIGNAL new_inclk : STD_LOGIC;
  SIGNAL start_inclk : STD_LOGIC;
  SIGNAL clklock_rising_edge_count : INTEGER := 0 ;
  SIGNAL clklock_cycle : TIME ;
  SIGNAL clklock_duty_cycle : TIME ;
  SIGNAL clklock_last_rising_edge : TIME ;
  SIGNAL clklock_last_falling_edge : TIME ;
  SIGNAL clock_count : INTEGER := -1 ;
  SIGNAL clklock_lock : BOOLEAN := TRUE;
-- pragma translate_on

BEGIN

-- pragma translate_off
edge_count: PROCESS
     VARIABLE expected_cycle : REAL;
     VARIABLE real_cycle : REAL;
BEGIN

  WAIT UNTIL (inclk'EVENT AND inclk='1');	   	-- Detect First Edge
     clklock_rising_edge_count <= clklock_rising_edge_count +1;
     clklock_last_rising_edge <= NOW;
     IF clklock_rising_edge_count = 0 THEN		-- at 1st rising edge
	start_inclk <= inclk;
     ELSE
     IF clklock_rising_edge_count = 1 THEN		-- at 2nd rising edge
	clklock_cycle <= NOW - clklock_last_rising_edge;	-- calculate period
	expected_cycle := 1000.0 / input_frequency);
	real_cycle := REAL( (NOW - clklock_last_rising_edge) / 1 ns);
	IF ( real_cycle < (expected_cycle - 1.0)  OR
	     real_cycle > (expected_cycle + 1.0) ) THEN
  	      	ASSERT ( (expected_cycle - 1.0) <= real_cycle AND
			  real_cycle <= (expected_cycle + 1.0) )
  		REPORT " Input_Frequency Violation "
  		SEVERITY WARNING;
  		clklock_lock <= FALSE;
              END IF;
     	IF ( (NOW - clklock_last_falling_edge) /= clklock_duty_cycle ) THEN
	      	ASSERT (NOW - clklock_last_falling_edge) = clklock_duty_cycle
		REPORT " Duty Cycle Violation "
		SEVERITY WARNING;
		clklock_lock <= FALSE;
     	END IF;
     ELSE
     IF ( (NOW - clklock_last_rising_edge) /= clklock_cycle ) THEN
      	ASSERT (NOW - clklock_last_rising_edge) = clklock_cycle
	REPORT " Cycle Violation "
	SEVERITY WARNING;
	clklock_lock <= FALSE;
     END IF;
     END IF;
     END IF;


  WAIT UNTIL (inclk'EVENT AND inclk='0');	  	-- Detect Secound Edge
     IF clklock_rising_edge_count = 1 THEN		-- at 1st falling edge
						 	-- Calculate new 1/2 Cycle
	clklock_half_period <= (NOW - clklock_last_rising_edge)/clockboost;
	clklock_duty_cycle <= NOW - clklock_last_rising_edge;	-- calculate duty cycle
	clklock_last_falling_edge <= NOW;
     ELSE
     IF ( (NOW - clklock_last_rising_edge) /= clklock_duty_cycle ) THEN
	ASSERT (NOW - clklock_last_rising_edge) = clklock_duty_cycle 
	REPORT " Duty Cycle Violation "
	SEVERITY WARNING;
	clklock_lock <= FALSE;
     END IF;
     END IF;

END PROCESS edge_count;

toggle: PROCESS
BEGIN

  WAIT ON clklock_rising_edge_count;
     IF clklock_rising_edge_count > 2 THEN
	FOR i IN 1 TO 2*clockboost LOOP	                -- Count the new clock edges
		clock_count <= clock_count + 1;
		WAIT FOR clklock_half_period;		
	END LOOP;
     ELSE
	clock_count <= 0;
     END IF;
END PROCESS toggle;


gen_outclk: PROCESS		         		-- Generate new clock
  VARIABLE outclk_zd : STD_LOGIC;
BEGIN

    IF ( clock_count <= 0 OR clklock_lock = FALSE ) THEN
	-- handle falling edge first case
	outclk <= '0';					-- avoid 'U'
 	outclk_zd := '0' ;
    END IF;
  WAIT ON clock_count, clklock_lock;
    IF ( clock_count = 0 OR clklock_lock = FALSE ) THEN
 	outclk_zd := '0' ;
    ELSE
    IF clock_count = 1 THEN
	new_inclk <= NOT start_inclk;
	outclk_zd := start_inclk ;
    ELSE
     	new_inclk <= NOT new_inclk;
     	outclk_zd := new_inclk ;
    END IF;
    END IF;

  outclk <= outclk_zd AFTER tpd;
END PROCESS gen_outclk;	
   
-- pragma translate_on
END behavior;

