/*******************************************************************
 * 
 * General Purpose IO (GPI/GPO) Simulatuon Model
 * 
 * GPIO are an unsychronised communication mechanism - literally wires.
 * The wires may or may not be registered.
 * 
 * SIMULATION
 *
 * The software analogue of a wire is a shared variable. However, while
 * it is OK for a hardware process to sit in a loop continuously performing
 * a combinatorial operation on a wire value (or waiting) this does not
 * map well into software.
 *
 * Two simulation models for wires are supplied. The threaded model
 * allows tasks to switch each time the value on the wire is changed or
 * tested. The threaded model does not. If you are using a GPIO and wire
 * to control how threads interact use the threaded model. If control is
 * described elsewhere (bus, handshaking channel, message passing) use the
 * non-threaded version.
 * 
 *		alt_wire_channel_threaded
 *		alt_wire_channel_nothreads
 *
 * The threaded model causes the calling process to yield after setting
 * and before reading the value of a wire
 * 
 * See the examples directory for an example of use.
 * 
 * Do not include this file directly - use alt_cusp.h
 * 
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif

#ifndef __ALT_GPIO_CHANNEL_H
#define __ALT_GPIO_CHANNEL_H



/*******************************************************************
 *  Model for wire channel interface
 ******************************************************************/
 
template <class T> class alt_wire_channel : public sc_channel {
    public:
        virtual ALT_GPO<T> &getWriter() = 0;
        virtual ALT_GPI<T> &makeNewReader() = 0;
        virtual ~alt_wire_channel() {};
        alt_wire_channel(sc_module_name channelName):sc_channel(channelName){}
    private:
        alt_wire_channel( const alt_wire_channel<T>& );
        alt_wire_channel<T>& operator = ( const alt_wire_channel<T>& );
};



/*******************************************************************
 *  Threaded implementation of alt_wire_channel interface
 ******************************************************************/
 
template <class T> class alt_wire_channel_threaded : public alt_wire_channel<T>{
    private:

        class _channel_writer : public ALT_GPO<T> {
            alt_wire_channel_threaded<T> &_channel;
            void (*_yield)();
        public:
		    virtual void setValue(const T& val) 
		    	{ _channel.channelValue = val; _channel.yield(); };
		    virtual const T& currentValue() 
		    	{ return _channel.channelValue; };
            _channel_writer(alt_wire_channel_threaded<T> &channel) 
            	: ALT_GPO<T>(), _channel(channel) {};
        };

        
        class _channel_reader : public ALT_GPI<T> {
            alt_wire_channel_threaded<T> &_channel;
            T _currentValue;
            _channel_reader *_next;
        public:
		    virtual const T&  nextValue() { _channel.yield(); _currentValue = _channel.channelValue; return _currentValue; };

		    virtual const T& currentValue() { return _currentValue; };
            _channel_reader(alt_wire_channel_threaded<T> &channel, _channel_reader *head) 
                : ALT_GPI<T>(), _channel(channel), _next(head) {};

            virtual ~_channel_reader() { delete _next; };
        };


        _channel_writer _writer;
        _channel_reader *_head;
            
    public:
        T channelValue;
   		virtual void yield() { wait(0, SC_NS); };               // this is an attempt at pthread_yield()

        virtual ALT_GPO<T> &getWriter() { return _writer; };
        virtual ALT_GPI<T> &makeNewReader() {
            _head = new _channel_reader(*this,_head); 
            return *_head;
        };

        alt_wire_channel_threaded(sc_module_name channelName) 
            : alt_wire_channel< T >(channelName), _writer(*this), _head(NULL)
		{
		}

        alt_wire_channel_threaded() 
            : alt_wire_channel< T >(sc_gen_unique_name( "wire_channel" )), _writer(*this), _head(NULL)
		{
		}

        virtual ~alt_wire_channel_threaded() { delete _head; };
};


/*******************************************************************
 *  No-threads implementation of alt_wire_channel interface
 ******************************************************************/

template <class T> class alt_wire_channel_nothreads : public alt_wire_channel_threaded<T> {
    public:
   		virtual void yield() {  };               

        alt_wire_channel_nothreads(sc_module_name channelName) 
            : alt_wire_channel_threaded< T >(channelName)
		{
		};

        alt_wire_channel_nothreads() 
            : alt_wire_channel_threaded< T >(sc_gen_unique_name( "wire_channel" ))
		{
		};

};


#endif
