/*******************************************************************
 * 
 * Avalon Bus Model
 * 
 * SIMULATION
 * 
 * A simple bus functional model capable of simulating single word
 * reads/writes across a bytes addressed bus is provided.
 * 
 * Features:
 * 		all addresses are BYTE addresses
 * 		an address map is maintained for each master
 * 		a default address map is maintained for the bus
 * 		transactions are made using sc_int's of undefined width
 *      (i.e. transactions may be of any size between 1 and 63 bits)
 * 
 * The following classes are provided
 *		alt_avalon_bfm
 * 		alt_avalon_simple_master
 *      ALT_AVALON_MEMMAP_SLAVE (in alt_avalon.h)
 * 
 * To use the model the bus functional model (bfm) must be instantiated.
 * Masters and slave should be instantiated and added to the bfm.
 * Slaves must be assigned an address using the mapSlave() function.
 * 
 * See the examples directory for an example of use.
 *  
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 * 
 * 
 * 
 *******************************************************************/
 

#ifndef __ALT_CUSP_H
#include "alt_cusp.h"
#endif


#ifndef __ALT_AVALON_BUS_H
#define __ALT_AVALON_BUS_H

class alt_avalon_master;
class alt_avalon_slave;

class alt_avalon_bus_matrix : public sc_channel  {
	private:
		friend class alt_avalon_slave;
		friend class alt_avalon_master;
		virtual void addSlave(alt_avalon_slave &slave) = 0;
		virtual void addMaster(alt_avalon_master &master) = 0;

	public:
		virtual void yield() = 0;

        // NB all addresses are BYTE ADDRESSES
        // ------------------------------------
		virtual void mapSlave(alt_avalon_slave &slave, unsigned int firstAddress, unsigned int lastAddress) = 0;
		virtual void mapSlave(alt_avalon_slave &slave, alt_avalon_master &master, unsigned int firstAddress, unsigned int lastAddress) = 0;
		
		virtual void write(alt_avalon_master &master, unsigned int address, const sc_int_base &data) = 0;
		virtual sc_int_base read(alt_avalon_master &master, unsigned int address) = 0;
		
		alt_avalon_bus_matrix(sc_module_name busName):sc_channel(busName){};
		
		virtual ~alt_avalon_bus_matrix() {};
		
};


class alt_avalon_master : virtual public sc_interface {
	public:
		virtual int getAddressWidth() = 0;
		virtual int getDataWidth() = 0;
		alt_avalon_master(alt_avalon_bus_matrix &bus) {
			bus.addMaster(*this);
		}
		virtual ~alt_avalon_master() {};
};

class alt_avalon_slave : virtual public sc_interface {
	public:
		virtual int getAddressWidth() = 0;
		virtual int getDataWidth() = 0;
		
        // NB all addresses are BYTE ADDRESSES
        // ------------------------------------
		virtual const sc_int_base &acceptBusRead(unsigned int address) = 0;
		virtual void acceptBusWrite(unsigned int address, const sc_int_base &data) = 0;
		
		alt_avalon_slave(alt_avalon_bus_matrix &bus) {
			bus.addSlave(*this);
		}
		virtual ~alt_avalon_slave() {};
};

// ===============================================================================
// ===============================================================================
// ===============================================================================
// ===============================================================================

class alt_avalon_bfm : public alt_avalon_bus_matrix {
	
	private:
	
		struct _AddressMapEntry {
			alt_avalon_slave *slave;
			unsigned int firstAddress;
			unsigned int lastAddress;
		};
		
		struct _AddressMap {
			alt_avalon_master *master; // may be null -> default
			sc_pvector<_AddressMapEntry*> entries;
		};
		
		sc_pvector< _AddressMap* > _decodeTable;
	
		alt_avalon_slave *_slave;
		alt_avalon_master *_master;
	
		virtual void addSlave(alt_avalon_slave &slave) 
		{ 
		};
		
		virtual void addMaster(alt_avalon_master &master) 
        {
            _AddressMap *masterMap = new _AddressMap();
            masterMap->master = &master;
            _decodeTable.push_back(masterMap);
		};


        void _addMapEntry ( _AddressMap *map, alt_avalon_slave *slave, unsigned int firstAddress, unsigned int lastAddress)
        {
            _AddressMapEntry *mapEntry = new _AddressMapEntry();
            mapEntry->slave = slave;
            mapEntry->firstAddress = firstAddress;
            mapEntry->lastAddress = lastAddress;

            // fixme - check for overlap

            map->entries.push_back(mapEntry);
        };

        
        _AddressMap *_findAddressMap(alt_avalon_master *master)
        {
			for (int i=0;i<_decodeTable.size();++i) {
				_AddressMap *map = _decodeTable[i];
				if (map->master == master) return map;
			}

            return NULL;
        };

        _AddressMapEntry *_findMapEntry(_AddressMap *map, unsigned int address)
        {
            for (int i=0;i<map->entries.size();i++) {
                _AddressMapEntry *entry = map->entries[i];
                if (entry->firstAddress<=address && entry->lastAddress>=address) return entry;
            }
            return NULL;
        }

        _AddressMapEntry *_findMapEntry(alt_avalon_master *master, unsigned int address)
        {
            _AddressMapEntry *slaveEntry = NULL;

            _AddressMap *masterMap = _findAddressMap(master);
            if (masterMap != NULL) slaveEntry = _findMapEntry(masterMap, address);

            if (slaveEntry == NULL)  {
                _AddressMap *commonMap  = _findAddressMap(NULL);
                slaveEntry = _findMapEntry(commonMap, address);
            }

            if (slaveEntry == NULL) {
                fprintf(stderr, "Master tries to access unmapped address 0x%x\n", address);
                exit(1);
            }

            return slaveEntry;
        }

	public:
		virtual void yield() { wait(0, SC_NS); }; // this is an attempt at pthread_yield()
	
// ======================================================		


	
		virtual void mapSlave(alt_avalon_slave &slave, 
						unsigned int firstAddress, unsigned int lastAddress) 
		{
            _AddressMap *commonMap = _findAddressMap(NULL);
            _addMapEntry (commonMap, &slave, firstAddress, lastAddress);
			
		};
		
		virtual void mapSlave(alt_avalon_slave &slave, alt_avalon_master &master, 
				unsigned int firstAddress, unsigned int lastAddress) 
		{
            _AddressMap *masterMap = _findAddressMap(&master);
            _addMapEntry (masterMap, &slave, firstAddress, lastAddress);

		};

// ======================================================		
		virtual void write(alt_avalon_master &master, 
							unsigned int address, const sc_int_base &data) 
		{
            _AddressMapEntry *slaveEntry = _findMapEntry(&master, address);
            unsigned long effectiveAddress = address - slaveEntry->firstAddress;
			slaveEntry->slave->acceptBusWrite(effectiveAddress,data);
		};
		
		virtual sc_int_base read(alt_avalon_master &master, unsigned int address)
		{
            _AddressMapEntry *slaveEntry = _findMapEntry(&master, address);
            unsigned long effectiveAddress = address - slaveEntry->firstAddress;
			return slaveEntry->slave->acceptBusRead(effectiveAddress);
		};
// ======================================================		
		
		alt_avalon_bfm(sc_module_name busName) 
            : alt_avalon_bus_matrix(busName),  _slave(NULL), _master(NULL)
		{
			_AddressMap *commonMap = new _AddressMap();
			commonMap->master = NULL;
			_decodeTable.push_back ( commonMap ); 
		}

        alt_avalon_bfm() 
            : alt_avalon_bus_matrix(sc_gen_unique_name( "av_bus" )), _slave(NULL), _master(NULL)
		{
			_AddressMap *commonMap = new _AddressMap();
			commonMap->master = NULL;
			_decodeTable.push_back ( commonMap ); 
		}
		
		
		virtual ~alt_avalon_bfm() {};
		
};

// ===============================================================================



template <int DATA_WIDTH, int BYTE_ADDR_WIDTH> class alt_avalon_simple_master : public alt_avalon_master {
	private:
		alt_avalon_bus_matrix &_bus;
		int _effectiveWidthBits;

	public:
		typedef sc_int<DATA_WIDTH> DATA_TYPE_S;
		typedef sc_uint<DATA_WIDTH> DATA_TYPE_U;
	
		virtual int getAddressWidth() { return BYTE_ADDR_WIDTH; };
		virtual int getDataWidth() { return _effectiveWidthBits; }
		
		virtual DATA_TYPE_S busReadSI(unsigned int address) 
		{
			_bus.yield();
			return (DATA_TYPE_S) _bus.read(*this, address); 
		};
		
		virtual void busWriteSI(unsigned int address, const DATA_TYPE_S data) 
		{ 
			_bus.write(*this, address, data); 
		} ;



		virtual DATA_TYPE_U busReadUI(unsigned int address) 
		{
			_bus.yield();
			return (DATA_TYPE_U) _bus.read(*this, address); 
		};
		
		virtual void busWriteUI(unsigned int address, const DATA_TYPE_U data) 
		{ 
			_bus.write(*this, address, (DATA_TYPE_S)data); 
		} ;

		
		alt_avalon_simple_master(alt_avalon_bus_matrix &bus) 
			: alt_avalon_master(bus), _bus(bus) {};
			
		virtual ~alt_avalon_simple_master()  {
            _effectiveWidthBits = 8;
            while (_effectiveWidthBits < DATA_WIDTH) _effectiveWidthBits *= 2;
        };
};




#endif // __ALT_AVALON_BUS_ H


