/*******************************************************************
 * 
 * Atlantic Channel  
 * 
 * SIMULATION
 * 
 * Atlantic channels are characterised as as either a SOURCE (producer)
 * or a SINK (consumer of data). Data passes across Atlantic channels in
 * a series of packets. Packets may range in size from 1 word to N words.
 * 
 * Atlantic channels are point-to-point communication channels with handshaking.
 * 
 * To assist simulation we provide three simulation models:
 * 
 * 	   alt_atlantic_point_to_point_channel
 * 	   alt_atlantic_file_to_point_channel
 *     alt_atlantic_point_to_file_channel
 * 
 * The _point_to_point_ simulation model is the simplest and may be used to
 * connect two modules that have ALT_ATLANTIC_SINK/_SOURCE ports.
 * 
 * The _file_to_point_ model takes in its constructor a file name and uses this
 * to generate data for the channel. It may be connected to an ATLANTIC_SINK.
 * 
 * The _point_to_file_ model takes in its constructor a file name and uses this
 * as the destination for any data placed in the channel. It may be connected
 * to an ATLANTIC_SOURCE
 * 
 * 
 * See the examples directory for an example of use.
 *  
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 * 
 * 
 * 
 *******************************************************************/

#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif


#ifndef __ALT_ATLANTIC_CHANNEL_H
#define __ALT_ATLANTIC_CHANNEL_H


template <class T> 
class alt_atlantic_channel : public sc_prim_channel 
{
	public:
		virtual ALT_ATLANTIC_SOURCE<T> &getSource() = 0;
		virtual ALT_ATLANTIC_SINK<T> &getSink() = 0;

		alt_atlantic_channel() : sc_prim_channel(sc_gen_unique_name("atlantic")) {}
		alt_atlantic_channel(const char * channelName) : sc_prim_channel(channelName) {}

		virtual ~alt_atlantic_channel() {}
};




// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
//
//              POINT-TO-FILE CHANNEL MODEL
//
// ==============================================================================================================================
// ==============================================================================================================================

template <class T> class alt_atlantic_point_to_file_channel : public alt_atlantic_channel<T>
{
private:
	class alt_atlantic_channel_source : public ALT_ATLANTIC_SOURCE<T>
	{
	private:
		ofstream _sinkFile;
		const char *_destinationFileName;
		bool _oneEntryPerLine;
    bool _noFile;

	public:
		virtual void write(const T & data)
		{
			_sinkFile << data;
			if (_oneEntryPerLine)
				_sinkFile << endl;
			_sinkFile.flush();                
		}

		virtual bool hasSpaceAvail(void)
		{
      if (_noFile == true)
      {
        return false;
      }
      else
      {
			  return true;
      }
		}

		alt_atlantic_channel_source(const char *channelName, const char *destinationFileName) 
		{
			_destinationFileName = destinationFileName;
			_oneEntryPerLine = true;
      _noFile = true;
			_sinkFile.open(_destinationFileName, ios::out | ios::trunc );
			if (!_sinkFile.is_open())
			{ 
				cout << "Failed to open file : "<< _destinationFileName << endl;
			}
      else
      {
        _noFile = false;
      }
		}

	};

	alt_atlantic_channel_source _source;

public:

	const char *_destinationFileName;
	const char *_channelName;

	ALT_ATLANTIC_SOURCE<T > &getSource() { return _source; };
	ALT_ATLANTIC_SINK<T > &getSink() 
	{ 
		cout <<"ERROR: Trying to get sink end of a channel to file"<<endl; 
    fflush(stdout);
		throw new alt_exception_config(); 
	}

	alt_atlantic_point_to_file_channel(const char * channelName, const char *destinationFileName) 
	  : alt_atlantic_channel< T >(channelName), _source(channelName, destinationFileName),
	    _destinationFileName(destinationFileName), _channelName(channelName)
	{
	}

	virtual ~alt_atlantic_point_to_file_channel() {};
	
};

// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
//
//              FILE-TO-POINT CHANNEL MODEL
//
// ==============================================================================================================================
// ==============================================================================================================================

template <class T > class alt_atlantic_file_to_point_channel : public alt_atlantic_channel< T >
{
private:	
	class alt_atlantic_channel_sink : public ALT_ATLANTIC_SINK<T>
	{
	private:

		ifstream _sourceFile;
		const char *_sourceFileName;
    bool  _noFile;

	public:
		virtual T read()
		{ 
			if ( EOF == _sourceFile.peek() )
			{
        while(1)
        {
          ::wait(1,SC_MS);
        }
			}
			T data;
			_sourceFile >> data;
			return data;
		}

		virtual bool hasDataAvail(void)
		{
      if (_noFile == true){
        return false;
      }
			else if (_sourceFile.peek() == EOF){
				return false;
			}
			else{
				return true;
			}
		}
			
		alt_atlantic_channel_sink(const char *channelName, const char *sourceFileName) 
		{
      _noFile = true;
			_sourceFileName = sourceFileName;
			_sourceFile.open(_sourceFileName, ios::in );
			if (!_sourceFile.is_open())
			{
				cout << "Failed to open file : "<< _sourceFileName << endl;
			}
      else
      {
        _noFile = false;
      }
		}
	};

	alt_atlantic_channel_sink _sink;

public:

	const char *_sourceFileName;
	const char *_channelName;

	ALT_ATLANTIC_SOURCE<T > &getSource() 
	{ 
		cout <<"ERROR: Trying to get source end of a channel to file"<<endl; 
    fflush(stdout);
		throw new alt_exception_config(); 
	}

	ALT_ATLANTIC_SINK<T > &getSink() { return _sink; };

	alt_atlantic_file_to_point_channel(const char * channelName, const char *sourceFileName) 
	  : alt_atlantic_channel<T>(channelName), _sink(channelName, sourceFileName),
	    _sourceFileName(sourceFileName), _channelName(channelName)
	{
	}

};



// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
// ==============================================================================================================================
//
//              POINT-TO-POINT CHANNEL MODEL
//
// ==============================================================================================================================
// ==============================================================================================================================

template <class T > class alt_atlantic_point_to_point_channel : public alt_atlantic_channel<T>
{
	private:	
		class alt_atlantic_channel_sink : public ALT_ATLANTIC_SINK<T >
		{
		private:
			alt_atlantic_point_to_point_channel<T > &_channel;

		public:
			virtual T read() { return _channel.read(); }
			virtual bool hasDataAvail(void) { return _channel.hasDataAvail(); }
			alt_atlantic_channel_sink(alt_atlantic_point_to_point_channel<T> &channel) : _channel(channel) {}
		};
	
		class alt_atlantic_channel_source : public ALT_ATLANTIC_SOURCE<T >
		{
		private:
			alt_atlantic_point_to_point_channel<T > &_channel;

		public:
			virtual void write(const T & data) { _channel.write(data); }
			virtual bool hasSpaceAvail(void) { return _channel.hasSpaceAvail(); }
			alt_atlantic_channel_source(alt_atlantic_point_to_point_channel<T > &channel) : _channel(channel) {}
		};

	T * _fifo;
	unsigned int _fifoDepth;
	unsigned int _inptr;
	unsigned int _outptr;
	const char *_channelName;
	sc_event _inEvent, _outEvent;

	alt_atlantic_channel_sink _sink;
	alt_atlantic_channel_source _source;

public:
	ALT_ATLANTIC_SOURCE<T> &getSource() { return _source; };
	ALT_ATLANTIC_SINK<T> &getSink() { return _sink; };

	alt_atlantic_point_to_point_channel(unsigned int depth = 128) 
	  : alt_atlantic_channel<T>(sc_gen_unique_name("atlantic_channel")), _sink(*this), _source(*this)
	{
		_fifoDepth = depth;
		_fifo = new T[depth];
		_inptr = 0;
		_outptr = 0;
		_channelName = alt_atlantic_channel<T>::name();
	}

	alt_atlantic_point_to_point_channel(const char * channelName, unsigned int depth = 128) 
	  : alt_atlantic_channel<T>(channelName), _sink(*this), _source(*this)
	{
		_fifoDepth = depth;
		_fifo = new T[depth];
		_inptr = 0;
		_outptr = 0;
		_channelName = channelName;
	}

	T read()
	{
		while (_outptr == _inptr)
			wait(_inEvent);

		T data = _fifo[_outptr];
		_outptr = (_outptr + 1) % _fifoDepth;

		notify(_outEvent);

		return data;
	}

	void write(const T & data)
	{
		int next_inptr = (_inptr + 1) % _fifoDepth;

		while (next_inptr == _outptr)
			wait(_outEvent);

		_fifo[_inptr] = data;
		_inptr = next_inptr;

		notify(_inEvent);
	}

	bool hasDataAvail(void)
	{
		return _inptr != _outptr;
	}

	bool hasSpaceAvail(void)
	{
		int next_inptr = (_inptr + 1) % _fifoDepth;
		return next_inptr != _outptr;
	}
};
#endif
