/*******************************************************************
 * 
 * Register function units.
 * 
 * 
 * 
 * Abbreviations :
 *  W          = WIDTH
 *  L          = LATENCY
 *  R          = RESTART
 *  IW         = INTEGER WIDTH (FOR FIXED POINT)
 *  QM         = QUANTIZATION MODE(FOR FIXED POINT)
 *  OFM        = OVERFLOW MODE (FOR FIXED POINT)
 *  NSB        = NUMBER OF SATURATED BITS (USED WITH OFM)
 * 
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 *  (2) ALT_REG_FIXED, ALT_REG_UFIXED, and ATL_REG_BV are not
 *      currently supported.
 * 
 * 
 * Function Unit Models:
 * 
 * ALT_REG the following operations for simulation and synthesis :
 * (1)  Synchronous Loading
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 *       (e) Bit vector
 * (2)  Conditional loading
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 *       (a) Bit vector
 * (3)  Synchronous Clearing
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Bit vector
 * 
 *******************************************************************/

#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif

#ifndef __ALT_REG_H
#define __ALT_REG_H
//--------------------------------------------------------------------
// Register function unit and its named static operators
//--------------------------------------------------------------------
 
template <int W=16, int L=1, int R=1> 
class ALT_REG
{
public:
   //--------------------------------------------------------------------
   // Unsigned integer conditional loading
   //--------------------------------------------------------------------
   static sc_uint<W> cLdUI(
   						  sc_uint<W> nextVal,
                          sc_uint<W> currVal,
                          bool enable
                          )
   	{
    	if(enable)
    	{
    		return nextVal;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}
	
	
   //--------------------------------------------------------------------
   // Signed integer conditional loading
   //--------------------------------------------------------------------
   static sc_int<W> cLdSI(
   						 sc_int<W> nextVal,
                         sc_int<W> currVal,
                         bool enable
                         )
   	{
    	if(enable)
    	{
    		return nextVal;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}
 
   //--------------------------------------------------------------------
   // Unsigned fixed point conditional loading
   //--------------------------------------------------------------------
   static sc_ufix cLdUF(
   						 sc_ufix nextVal,
                         sc_ufix currVal,
                         bool enable
                         )
   	{
    	return cLdUF(nextVal, currVal, enable, 0);
	}

   //--------------------------------------------------------------------
   // Signed fixed point conditional loading
   //--------------------------------------------------------------------
   static sc_fix cLdSF(
   						 sc_fix nextVal,
                         sc_fix currVal,
                         bool enable
                         )
   	{
   		return cLdSF(nextVal, currVal, enable, 0);
	}

  
   //--------------------------------------------------------------------
   // Bit vector conditional loading
   //--------------------------------------------------------------------
   static sc_bv<W> cLdBV(
   					  sc_bv<W> nextVal,
                      sc_bv<W> currVal,
                      bool enable
                      )
   	{
    	if(enable)
    	{
    		return nextVal;
    	}	 
    	else
    	{
    		return currVal;
    	}
	} 
	
	
   //--------------------------------------------------------------------
   // Single precision conditional loading
   //--------------------------------------------------------------------
   static float cLdSP(
   						 float nextVal,
                         float currVal,
                         bool enable
                         )
   	{
    	if(enable)
    	{
    		return nextVal;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}

   //--------------------------------------------------------------------
   // Double precision conditional loading
   //--------------------------------------------------------------------
   static double cLdDP(
   						 double nextVal,
                         double currVal,
                         bool enable
                         )
   	{
   		if(enable)
    	{
    		return nextVal;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}
	
	  
   
   //--------------------------------------------------------------------
   // ld() functions (Synchronous load)
   //--------------------------------------------------------------------
	static sc_uint<W> ldUI(sc_uint<W> a)
   	{
    	return cLdUI(a, 0, true);	 
	}

  	static sc_int<W> ldSI(sc_int<W> a)
	{
		return cLdSI(a, 0, true);	 	 
	}
	
	static sc_bv<W> ldBV(sc_bv<W> a)
	{
		return cLdBV(a, 0, true);	  
   	}
 
	static sc_ufix ldUF(sc_ufix a)
	{
		return cLdUF(a, 0, true, 1);	 	 
	}

	static sc_fix ldSF(sc_fix a)
	{
		return cLdSF(a, 0, true, 1);	  
	}
	
	static float ldSP(float a)
	{
		return cLdSP(a, 0, true);	 	 
	}

	static double ldDP(double a)
	{
		return cLdDP(a, 0, true);	  
	}
	
	
	
   //--------------------------------------------------------------------
   // sClr() functions (Synchronous clear)
   //--------------------------------------------------------------------
   	static sc_uint<W> sClrUI()
   	{
    	return cLdUI(0, 0, true);	 
	}

  	static sc_int<W> sClrSI()
	{
		return cLdSI(0, 0, true);	 	 
	}
	
	static sc_bv<W> sClrBV()
	{
		return cLdBV(0, 0, true);	  
   	}
	
   	
private:
    static sc_ufix cLdUF(
   						 sc_ufix nextVal,
                         sc_ufix currVal,
                         bool    enable,
                         bool    iwlMode
                         )
   	{
   		int intWidthOfOut; 
    	
    	switch(iwlMode)
    	{
    		case 0 :  intWidthOfOut = currVal.iwl();
    				  break;
    		default :  intWidthOfOut = nextVal.iwl();
    	}
    	
    	if(enable)
    	{
    		sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
    		
    		result = nextVal;
    		
    		ALT_UTIL::warnIfNotEqualsUF(nextVal, result);
    		
    		return result;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}

   //--------------------------------------------------------------------
   // Signed fixed point conditional loading
   //--------------------------------------------------------------------
   static sc_fix cLdSF(
   						 sc_fix nextVal,
                         sc_fix currVal,
                         bool   enable,
                         bool   iwlMode
                         )
   	{
   		
   		int intWidthOfOut; 
    	
    	switch(iwlMode)
    	{
    		case 0 :  intWidthOfOut = currVal.iwl();
    				  break;
    		default :  intWidthOfOut = nextVal.iwl();
    	}
    	
    	if(enable)
    	{
    		sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
    		
    		result = nextVal;
    		
    		ALT_UTIL::warnIfNotEqualsUF(nextVal, result);
    		
    		return result;
    	}	 
    	else
    	{
    		return currVal;
    	}
	}

};


 /*******************************************************************
 * A number of convenience classes which declare a function unit and
 * declare and bind a variable to it are also provided.
 *  
 * These should generally not be used.
 * 
 * ALT_REG_UINT supports:
 * (1) All of ALT_REG operations for simulation and synthesis
 * (2) All of sc_uint operations for simulation
 * (3) The following sc_uint operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         +  -
 *      Assignment         =  +=  -=  
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_REG_INT supports:
 * (1) All of ALT_REG operations
 * (2) All of sc_int operations (Not all operations are synthesisable)
 * (3) The following sc_int operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         +  -
 *      Assignment         =  +=  -=  
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_REG_UFIXED supports:
 * (1) All of ALT_REG operations
 * (2) All of sc_ufixed operations for simulation
 * (3) The following sc_ufixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         +  -    
 *      Assignment         =  +=  -=     
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 * ALT_REG_FIXED supports:
 * (1) All of ALT_REG operations
 * (2) All of sc_fixed operations for simulation
 * (3) The following sc_fixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         +  -    
 *      Assignment         =  +=  -=     
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 * ALT_AU_BV supports:
 * (1) All of ALT_AU operations
 * (2) All of sc_bv operations for simulation
 * (3) The following sc_bv operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  <<     (as wire or combinational operations)
 *      Assignment         =
 *      Bit Select        []
 *      Part Select       range()    
 *      Concatenation     (,)
 *      Reduction         and_reduce()  or_reduce()  xor_reduce()(as combinational operations)
 *
 * 
 * 
 * 
 *  
 *******************************************************************/



//--------------------------------------------------------------------
// Register function unit with an sc_int variable
//--------------------------------------------------------------------

template <int W=16, int L=1, int R=1> 
class ALT_REG_INT : public sc_int<W>, public ALT_REG<W, L, R>   
{
	#define CUSP_NAME ALT_REG_INT
	#define CUSP_TYPE_NAME ALT_REG_INT<W, L, R>
	#define CUSP_WRAP_SC_INT
	#define CUSP_WRAP_INT_SIZE W
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_int<W>( a ) {}

	#undef CUSP_WRAP_INT_SIZE
	#undef CUSP_WRAP_SC_INT
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};
 
 
//--------------------------------------------------------------------
// Register function unit with an sc_uint variable
//--------------------------------------------------------------------

template <int W=16, int L=1, int R=1> 
class ALT_REG_UINT : public sc_uint<W>, public ALT_REG<W, L, R>   
{
	#define CUSP_NAME ALT_REG_UINT
	#define CUSP_TYPE_NAME ALT_REG_UINT<W, L, R>
	#define CUSP_WRAP_SC_UINT
	#define CUSP_WRAP_INT_SIZE W
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_uint<W>( a ) {}

	#undef CUSP_WRAP_INT_SIZE
	#undef CUSP_WRAP_SC_UINT
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};


//--------------------------------------------------------------------
// Register function unit with an sc_bv variable
//--------------------------------------------------------------------
 
template <int W=16, int L=1, int R=1>
class ALT_REG_BV : public sc_bv<W>, public ALT_REG<W, L, R>   
{
	#define CUSP_NAME ALT_REG_BV
	#define CUSP_TYPE_NAME ALT_REG_BV<W, L, R>
	#define CUSP_WRAP_SC_BV
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_bv<W>( a ) {}

	#undef CUSP_WRAP_SC_BV
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};


//--------------------------------------------------------------------
// Register function unit with an sc_fixed variable
//--------------------------------------------------------------------

template <int W=16, int L=1, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0> 
class ALT_REG_FIXED : public sc_fixed<W, IW, QM, OFM, NSB>, public ALT_REG<W, L, R>   
{
	#define CUSP_NAME ALT_REG_FIXED
	#define CUSP_TYPE_NAME ALT_REG_FIXED<W, L, R, IW, QM, OFM, NSB>
	#define CUSP_WRAP_SC_FIXED
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_fixed<W, IW, QM, OFM, NSB>( a ) {}

	#undef CUSP_WRAP_SC_FIXED
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};
 
 
//--------------------------------------------------------------------
// Register function unit with an sc_ufixed variable
//--------------------------------------------------------------------
 
template <int W=16, int L=1, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0> 
class ALT_REG_UFIXED : public sc_ufixed<W, IW, QM, OFM, NSB>, public ALT_REG<W, L, R>   
{
	#define CUSP_NAME ALT_REG_UFIXED
	#define CUSP_TYPE_NAME ALT_REG_UFIXED<W, L, R, IW, QM, OFM, NSB>
	#define CUSP_WRAP_SC_UFIXED
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_ufixed<W, IW, QM, OFM, NSB>( a ) {}

	#undef CUSP_WRAP_SC_UFIXED
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};


#endif
