/*******************************************************************
 * 
 * Multiplier function units.
 * 
 * 
 * 
 * Abbreviations :
 *  W          = WIDTH
 *  L          = LATENCY
 *  R          = RESTART
 *  IW         = INTEGER WIDTH (FOR FIXED POINT)
 *  QM         = QUANTIZATION MODE(FOR FIXED POINT)
 *  OFM        = OVERFLOW MODE (FOR FIXED POINT)
 *  NSB        = NUMBER OF SATURATED BITS (USED WITH OFM)
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 *  (2) ALT_AU_INT, ALT_AU_UINT, ALT_AU_FIXED and ALT_AU_UFIXED 
 *      are not currently supported
 * 
 * 
 * Function Unit Models:
 * 
 * ALT_MULT the following operations for simulation and synthesis :
 * (1)  Multiplication
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 *       (e) Unsigned X Signed integer
 *       (f) Signed X Unsigned integer
 *       (g) Unsigned X Signed fixed point
 *       (h) Signed X Unsigned fixed point
 * 
 *******************************************************************/
#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif


#ifndef __ALT_MULT_H
#define __ALT_MULT_H
//--------------------------------------------------------------------
// Multiplier function unit and its named static operators
//--------------------------------------------------------------------

template <int W=16, int L=2, int R=1> 
class ALT_MULT
{
public:
   //--------------------------------------------------------------------
   // mult() functions
   //--------------------------------------------------------------------
	static sc_uint<W * 2> multUI(const sc_uint<W> &a, const sc_uint<W> &b) 
	{
		return cMultUI(a, b); 	 
	}
	
    static sc_int<W * 2> multSI(const sc_int<W> &a, const sc_int<W> &b) 
	{
		return cMultSI(a, b);	
	}
   
    static sc_ufix multUF(const sc_ufix &a, const sc_ufix &b) 
	{
		return cMultUF(a, b); 
	}
   
	static sc_fix multSF(const sc_fix &a, const sc_fix &b) 
	{
		return cMultSF(a, b); 
	}
	
	// mixed signed/unsigned modes	
	static sc_int<W * 2> multSUI(const sc_int<W> &a, const sc_uint<W> &b) 
	{
		return cMultSUI(a, b);	
	}
	
	static sc_int<W * 2> multUSI(const sc_uint<W> &a, const sc_int<W> &b) 
	{
		return cMultSUI(b, a);	
	}
	
	static sc_fix multSUF(const sc_fix &a, const sc_ufix &b) 
	{
		return cMultSUF(a, b); 
	}
	
	static sc_fix multUSF(const sc_fix &a, const sc_fix &b) 
	{
		return cMultSUF(b, a); 
	}

private:
   //--------------------------------------------------------------------
   // Unsigned integer conditional multiplication
   //--------------------------------------------------------------------
	static sc_uint<W * 2> cMultUI(
	                             const sc_uint<W>      &a, 
	                             const sc_uint<W>      &b
	                             ) 
    {
		    sc_uint<W * 2> result = a * b;	
            return result; 	
	}


   //--------------------------------------------------------------------
   // Signed integer conditional multiplication
   //--------------------------------------------------------------------
    static sc_int<W * 2> cMultSI(
	                             const sc_int<W>      &a, 
	                             const sc_int<W>      &b
	                             ) 
    {
		    sc_int<W * 2> result = a * b;	
            return result; 	
	}
	
   //--------------------------------------------------------------------
   // Signed X Unsigned integer  conditional multiplication
   //--------------------------------------------------------------------
    static sc_int<W * 2> cMultSUI(
	                             const sc_int<W>      &a, 
	                             const sc_uint<W>      &b
	                             ) 
    {
		    sc_int<W * 2> result = a * b;	
            return result; 	
	}
	
	
   //--------------------------------------------------------------------
   // Unsigned fixed conditional multiplication
   // If the word length of either "a" or "b" is greater than the input
   // bit width of the fuction unit "W", the LSBs will be deleted.
   // If the integer word length is greater than the word length, integer
   // word length will be set to word length.
   // If the integer word length is negative program will terminate. 
   //--------------------------------------------------------------------
	static sc_ufix cMultUF(
	                       const sc_ufix   &a, 
	                       const sc_ufix   &b
	                      ) 
    {            
            sc_ufix aTrn(W, a.iwl(), SC_TRN, SC_WRAP, 0);
            sc_ufix bTrn(W, b.iwl(), SC_TRN, SC_WRAP, 0);
            aTrn = a;
            bTrn = b;
            
            ALT_UTIL::warnIfNotEqualsUF(a, aTrn);
            ALT_UTIL::warnIfNotEqualsUF(b, bTrn);
                      
            sc_ufix result(W * 2, aTrn.iwl() + bTrn.iwl(), SC_TRN, SC_WRAP, 0);
		    
		    result = aTrn * bTrn;	
            return result; 	
	}
	

   //--------------------------------------------------------------------
   // Signed fixed conditional multiplication
   // If the word length of either "a" or "b" is greater than the input
   // bit width of the fuction unit "W", the LSBs will be deleted.
   // If the integer word length is greater than the word length, integer
   // word length will be set to word length.
   // If the integer word length is negative program will terminate. 
   //--------------------------------------------------------------------
	static sc_fix cMultSF(
	                      const sc_fix   &a, 
	                      const sc_fix   &b
	                     ) 
    {
            sc_fix aTrn(W, a.iwl(), SC_TRN, SC_WRAP, 0);
            sc_fix bTrn(W, b.iwl(), SC_TRN, SC_WRAP, 0);
            aTrn = a;
            bTrn = b;
            
            ALT_UTIL::warnIfNotEqualsSF(a, aTrn);
            ALT_UTIL::warnIfNotEqualsSF(b, bTrn);
                      
            sc_fix result(W * 2, aTrn.iwl() + bTrn.iwl(), SC_TRN, SC_WRAP, 0);
		    
		    result = aTrn * bTrn;	
            return result; 	 		
	}
	
   //--------------------------------------------------------------------
   // Signed x Unsigned fixed conditional multiplication
   // If the word length of either "a" or "b" is greater than the input
   // bit width of the fuction unit "W", the LSBs will be deleted.
   // If the integer word length is greater than the word length, integer
   // word length will be set to word length.
   // If the integer word length is negative program will terminate. 
   //--------------------------------------------------------------------
	static sc_fix cMultSUF(
	                      const sc_fix   &a, 
	                      const sc_ufix   &b
	                     ) 
    {
            sc_fix aTrn(W, a.iwl(), SC_TRN, SC_WRAP, 0);
            sc_ufix bTrn(W, b.iwl(), SC_TRN, SC_WRAP, 0);
            aTrn = a;
            bTrn = b;
            
            ALT_UTIL::warnIfNotEqualsSF(a, aTrn);
            ALT_UTIL::warnIfNotEqualsSF(b, bTrn);
                      
            sc_fix result(W * 2, aTrn.iwl() + bTrn.iwl(), SC_TRN, SC_WRAP, 0);
		    
		    result = aTrn * bTrn;	
            return result; 	 		
	}
	
};



 /*******************************************************************
 * A number of convenience classes which declare a function unit and
 * declare and bind a variable to it are also provided.
 *  
 * These should generally not be used.
 * 
 * 
 * ALT_MULT_UINT supports:
 * (1) All of ALT_MULT operations for simulation and synthesis
 * (2) All of sc_uint operations for simulation
 * (3) The following sc_uint operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         *
 *      Assignment         =  *=  
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_MULT_INT supports:
 * (1) All of ALT_MULT operations
 * (2) All of sc_int operations (Not all operations are synthesisable)
 * (3) The following sc_int operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         *
 *      Assignment         =  *=
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_MULT_UFIXED supports:
 * (1) All of ALT_MULT operations
 * (2) All of sc_ufixed operations for simulation
 * (3) The following sc_ufixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         *    
 *      Assignment         =  *=
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 * ALT_MULT_FIXED supports:
 * (1) All of ALT_MULT operations
 * (2) All of sc_fixed operations for simulation
 * (3) The following sc_fixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         *    
 *      Assignment         =  *=     
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 *******************************************************************/




//--------------------------------------------------------------------
// Multiplier function unit with an sc_int variable
//--------------------------------------------------------------------

template <int W=16, int L=2, int R=1>
class ALT_MULT_INT : public sc_int< W * 2 >, public ALT_MULT<W, L, R>   
{
	#define CUSP_NAME ALT_MULT_INT
	#define CUSP_TYPE_NAME ALT_MULT_INT<W, L, R>
	#define CUSP_WRAP_SC_INT
	#define CUSP_WRAP_INT_SIZE (2*W)
	
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_int<W*2>( a ) {}

	#undef CUSP_WRAP_INT_SIZE
	#undef CUSP_WRAP_SC_INT
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};
 
 
//--------------------------------------------------------------------
// Multiplier function unit with an sc_uint variable
//--------------------------------------------------------------------

template <int W=16, int L=2, int R=1> 
class ALT_MULT_UINT : public sc_uint<W * 2>, public ALT_MULT<W, L, R>   
{
	#define CUSP_NAME ALT_MULT_UINT
	#define CUSP_TYPE_NAME ALT_MULT_UINT<W, L, R>
	#define CUSP_WRAP_SC_UINT
	#define CUSP_WRAP_INT_SIZE (2*W)

	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_uint<W * 2>( a ) {}

	#undef CUSP_WRAP_INT_SIZE
	#undef CUSP_WRAP_SC_UINT
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};


//--------------------------------------------------------------------
// Multiplier function unit with an sc_fixed variable
//--------------------------------------------------------------------
 
template <int W=16, int L=2, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0>
class ALT_MULT_FIXED : public sc_fixed<W*2, IW*2, QM, OFM, NSB*2>, 
                       public ALT_MULT<W, L, R>   
{
	#define CUSP_NAME ALT_MULT_FIXED
	#define CUSP_TYPE_NAME ALT_MULT_FIXED<W, L, R, IW, QM, OFM, NSB>
	#define CUSP_WRAP_SC_FIXED
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_fixed<W*2, IW*2, QM, OFM, NSB*2>( a ) {}

	#undef CUSP_WRAP_SC_FIXED
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};
 
 
//--------------------------------------------------------------------
// Multiplier function unit with an sc_ufixed variable
//--------------------------------------------------------------------
 
template <int W=16, int L=2, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0>
class ALT_MULT_UFIXED : public sc_ufixed<W*2, IW*2, QM, OFM, NSB*2>, 
                       public ALT_MULT<W, L, R>   
{
	#define CUSP_NAME ALT_MULT_UFIXED
	#define CUSP_TYPE_NAME ALT_MULT_UFIXED<W, L, R, IW, QM, OFM, NSB>
	#define CUSP_WRAP_SC_UFIXED
	#include "cusp/alt_wrap.h"

	CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_ufixed<W*2, IW*2, QM, OFM, NSB*2>( a ) {}

	#undef CUSP_WRAP_SC_UFIXED
	#undef CUSP_TYPE_NAME
	#undef CUSP_NAME
};
#endif
