/*******************************************************************
 * 
 * General Purpose IO (GPI/GPO) function units.
 * 
 * GPIO are an unsychronised communication mechanism - literally wires.
 * The wires may or may not be registered.
 * 
 * Abbreviations :
 *  T          = A TYPE (e.g. short, char, sc_int<6>, etc)
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 * 
 * 
 * Function Unit Models:
 * 
 * ALT_GPI<T>  
 *    - A general purpose input that has type T.
 *   
 * 
 * ALT_GPI the following operations for simulation and synthesis :
 *   ->nextValue()
 * 			    read a new value from the input port 
 *   ->currentValue()
 *              read a value from the input port
 * 
 *  Calls to nextValue() are guaranteed to fall in different 
 *  clock cycles and be scheduled in order of writing.
 *  
 *  Calls to currentValue() may be scheduled in any clock cycle 
 *  from (and including) the instant at which the last call to nextValue()
 *  was scheduled until the instant before the next call to nextValue()
 *  is scheduled. The order in which currentValue() calls are scheduled
 *  is not specified.
 * 
 *  NB You probably want to use nextValue() to read in a series of 
 *  samples.
 * 
 * ALT_GPO<T>  
 *    - A general purpose input that has type T.
 *   
 * 
 * ALT_GPO the following operations for simulation and synthesis :
 *   ->setValue(T)
 * 			    sets the new value of the output port
 *   ->currentValue()
 *              reads the current value from the output port
 * 
 *  Calls to settValue() are guaranteed to fall in different 
 *  clock cycles and be scheduled in order of writing.
 *  
 *  Calls to currentValue() may be scheduled in any clock cycle 
 *  from the instant at which the last call to setValue()
 *  was scheduled until the instant the next call to setValue()
 *  is scheduled. The order in which currentValue() calls are scheduled
 *  is not specified. If the output is unregisterd currentValue() may
 *  give an unspecified value.
 * 
 * 
 * 
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif

#ifndef __ALT_GPIO_H
#define __ALT_GPIO_H

/*******************************************************************
 *  Model for general purpose input interface
 ******************************************************************/

template <class T>  class ALT_GPI : virtual public sc_interface {
	public:
		virtual const T& nextValue() = 0;
		virtual const T& currentValue() = 0;
        virtual ~ALT_GPI() {};
        ALT_GPI() {};
    private:
        ALT_GPI( const ALT_GPI<T>& );
        ALT_GPI<T>& operator = ( const ALT_GPI<T>& ); 
};


/*******************************************************************
 *  Model for general purpose output interface
 ******************************************************************/
 
template <class T>  class ALT_GPO : virtual public sc_interface {
	public:
		virtual void setValue(const T& val) = 0;
		virtual const T& currentValue() = 0;
        virtual ~ALT_GPO() {};
        ALT_GPO() {};
    private:
        ALT_GPO( const ALT_GPO<T>& );
        ALT_GPO<T>& operator = ( const ALT_GPO<T>& );
};




#endif
