/*******************************************************************
 * 
 * Comparator function unit and its named static operators
 * 
 * 
 * 
 * Abbreviations:
 *  W = WIDTH
 *  L = LATENCY
 *  R = RESTART
 * 
 * 
 * Notes:
 *  (1) Do not include this file directly - use alt_cusp.h
 * 
 * 
 * Function Unit Models:
 * 
 * ALT_CMP the following operations for simulation and synthesis :
 * (1)  Equality
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * (2)  Inequality
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * (3)  Greater than
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * (4)  Less than
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * (5)  Greater than or equals
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * (6)  Less than or equals
 *       (a) Unsigned integer
 *       (b) Signed integer
 *       (c) Unsigned fixed point
 *       (d) Signed fixed point
 * 
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif

#ifndef __ALT_CMP_H
#define __ALT_CMP_H

template <int W=16, int L=1, int R=1> 
class ALT_CMP  
{

public:	

   //--------------------------------------------------------------------
   // Greater than methods
   //--------------------------------------------------------------------
	static bool gtUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a > b ;
	}
   
   
    static bool gtSI(sc_int<W> a, sc_int<W> b)
	{
		return a > b;
	}
	
	static bool gtUF(sc_ufix a, sc_ufix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
        
        ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned > bAligned;
	}
	
	static bool gtSF(sc_fix a, sc_fix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned > bAligned;
	}
	
	
   
   //--------------------------------------------------------------------
   // Less than methods
   //--------------------------------------------------------------------
	static bool ltUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a < b;
	}
	
	static bool ltSI(sc_int<W> a, sc_int<W> b)
	{
		return a < b;
	}
   
    static bool ltUF(sc_ufix a, sc_ufix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned < bAligned;
	}
   
    static bool ltSF(sc_fix a, sc_fix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned < bAligned;
	}
   
   
   //--------------------------------------------------------------------
   // Greater than or equals methods
   //--------------------------------------------------------------------
	static bool gteUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a >= b;
	}
	
	static bool gteSI(sc_int<W> a, sc_int<W> b)
	{
		return a >= b;
	}
   
    static bool gteUF(sc_ufix a, sc_ufix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned >= bAligned;
	}
   
    static bool gteSF(sc_fix a, sc_fix b)
	{		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned >= bAligned;
	}
   
   
   
   //--------------------------------------------------------------------
   // Less than or equals methods
   //--------------------------------------------------------------------
	static bool lteUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a <= b;
	}
	
	static bool lteSI(sc_int<W> a, sc_int<W> b)
	{
		return a <= b;
	}
   
    static bool lteUF(sc_ufix a, sc_ufix b)
	{
		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
        
        ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned <= bAligned;
	}
	
	static bool lteSF(sc_fix a, sc_fix b)
	{
		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned <= bAligned;
	}
	
   
   
   //--------------------------------------------------------------------
   // Equals methods
   //--------------------------------------------------------------------
	static bool eUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a == b;
	}
   
	static bool eSI(sc_int<W> a, sc_int<W> b)
	{
		return a == b;
	}
	
	static bool eUF(sc_ufix a, sc_ufix b)
	{
		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
        
        ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned == bAligned;
	}
	
	static bool eSF(sc_fix a, sc_fix b)
	{
		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned == bAligned;
	}
	
	
   //--------------------------------------------------------------------
   // Not-equals methods
   //--------------------------------------------------------------------
	static bool neUI(sc_uint<W> a, sc_uint<W> b)
	{
		return a != b;
	}
   
	static bool neSI(sc_int<W> a, sc_int<W> b)
	{
		return a != b;
	}
	
	static bool neUF(sc_ufix a, sc_ufix b)
	{
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
		
		ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
		
		return aAligned != bAligned;
	}
	
	static bool neSF(sc_fix a, sc_fix b)
	{		
		int intWidth = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix aAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidth, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
        
        ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
		
		return aAligned != bAligned;
	}
	
};
#endif
