/*******************************************************************
 * 
 * Avalon Bus (Master/Slave) function units.
 * 
 * The bus master/slave function units allow a processor to connect to a 
 * simple bus such as basic Avalon.
 * 
 * 
 * Abbreviations :
 *  None.
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 *  (2) The Avalon switch fabric is modelled as transfering data of type sc_uint<N>/sc_int<N>
 *      to cast an sc_uint of width N to a sc_fixed of width N with INT_BITS integer bits
 *
 *      i.e. if VALUE is an sc_uint<N>
 * 
 *       (sc_fixed<N,INT_BITS>) ( ( (sc_fixed<N,N>) VALUE )  >> (N-INT_BITS) )
 *
 *       #define TOFP(x) (sc_fixed<N,INT_BITS>)(((sc_fixed<N,N>)(x))>>(18-INT_BITS))
 * 
 * Function Unit Models:
 *  ALT_AVALON_MEMMAP_SLAVE <DATA_WIDTH, DEPTH>
 *     Description: An avalon slave implemented as a dual-port memory. One side is
 *     accessible to the avalon bus. The other provides a word addressed
 *     memory accessible by the processor.
 *     
 *     Operations supported for simulation and synthesis:
 *        (1) Memory Read
 *           (a) Unsigned integer
 *           (b) Signed integer  
 *   
 *    
 *  ALT_AVALON_MASTER<DATA_WIDTH, BYTE_ADDR_WIDTH>
 *     Operations supported for simulation and synthesis:
 *        (1) Bus Read
 *           (a) Unsigned integer
 *           (b) Signed integer  
 *        (2) Bus Write
 *           (a) Unsigned integer
 *           (b) Signed integer   
 *        (3) Bus Byte Write
 *           (a) Unsigned integer
 *           (b) Signed integer   
 * 
 * ALT_AVALON_MASTER_FIFO<DATA_WIDTH, BYTE_ADDR_WIDTH>
 *     Operations supported for simulation and synthesis:
 *        (1) Bus Read
 *           Description: To perform a bus read call busPostRead(address)
 *           followed by busCollectRead(). Any number of busPostReads can
 *           be issued before calling busCollectRead() provided the read
 *           FIFO depth is not exceeded. busCollectRead() will return data
 *           values in the order the busPostReads were issued.
 *           (a) Unsigned integer
 *           (b) Signed integer  
 *        (2) Bus Write
 *           (a) Unsigned integer
 *           (b) Signed integer   
 *        (3) Bus Byte Write
 *           (a) Unsigned integer
 *           (b) Signed integer
 *
 * 
 * ALT_AVALON_SINK<DATA_WIDTH, DEPTH, THRESHOLD>
 *     Description: An avalon write-only slave including a fifo. The Avalon
 *     streaming signal readyfordata is asserted when there are approximately
 *     THRESHOLD words available in the fifo. 
 *
 *     WIDTH default 16
 *     DEPTH default 32
 *     THRESHOLD default 20
 *
 * 
 *     Operations supported for simulation and synthesis:
 *        (1) read                                  [ read   ]
 *            Description: one word is read from the input fifo. If the 
 *            fifo is empty the program is suspended.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (2) conditional read                      [ cread   ]
 *            Description: one word is read from the input fifo if the condition 
 *            is true. If the condition is false the value returned is undefined. 
 *            If the fifo is empty the program is suspended.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (3) isEmpty                               [ isEmpty ]
 *            Returns true if there is data in the fifo.
 *            and non block reads to be paired to poll the status of the fifo:
 *            NB A conext switch will not occur automatically here. Programs busy waiting
 *            on isEmpty must add their own synchronisation.
 *            i.e.
 *            for (;;) {
 *               while (fifo->isEmpty()) {
 *                  ALT_CUSP_SYNTHESIS_OFF
 *                  task_switch.notify();
 *                  ALT_CUSP_SYNTHESIS_OFF
 *               }
 *               short data = fifo->read_nb();
 *            }
 *        (4) non blocking read                     [ nbread ]
 *            Description: one word is read from the input fifo if it has data otherwise
 *            the value returned is undefined. A non blocking read issued when the fifo is
 *            empty will not cause the fifo to be corrupted (underflow)
 *            NB A conext switch will not occur automatically here. Programs executing non-
 *            blocking reads must include their own sychronisation operations to ensure data 
 *            is inserted into the fifo.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (4) non blocking conditional read         [ nbcread ]
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *
 * ALT_AVALON_SOURCE<DATA_WIDTH, DEPTH, THRESHOLD>
 *     Description: An avalon read-only slave including a fifo. The Avalon
 *     streaming signal dataavailable is asserted when there are approximately
 *     THRESHOLD words available in the fifo. 
 *
 *     WIDTH default 16
 *     DEPTH default 32
 *     THRESHOLD default 20
 *
 * 
 *     Operations supported for simulation and synthesis:
 *        (1) write                                 [ write   ]
 *            Description: one word is written into the output fifo. If the 
 *            fifo is full the program is suspended.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (2) conditional write                      [ cwrite   ]
 *            Description: one word is written into the input fifo if the condition 
 *            is true. 
 *            If the fifo is full and the condition is true the program is suspended.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (3) isFull                                 [ isFull ]
 *            Returns true if there is no data in the fifo.
 *            and non block reads to be paired to poll the status of the fifo:
 *            NB A conext switch will not occur automatically here. Programs waiting on
 *            ifFull must add their own synchronisation.
 *            i.e.
 *            for (;;) {
 *               while (fifo->isFull()) {
 *                  ALT_CUSP_SYNTHESIS_OFF
 *                  task_switch.notify();
 *                  ALT_CUSP_SYNTHESIS_OFF
 *               }
 *               fifo->nbwrite(data);
 *            }
 *        (4) non blocking write                     [ nbwrite ]
 *            Description: one word is written into the input fifo if it has data otherwise
 *            the value returned is undefined. A non blocking read issued when the fifo is
 *            empty will not cause the fifo to be corrupted (underflow)
 *            NB A conext switch will not occur automatically here. Programs executing non-
 *            blocking write must include their own sychronisation operations to ensure data 
 *            is removed from the fifo.
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector
 *        (4) non blocking conditional write         [ nbcwrite]
 *            (a) Unsigned integer
 *            (b) Signed integer
 *            (c) Bit Vector
 *            (d) Logic Vector

  
    *
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif


#ifndef __ALT_AVALON_H
#define __ALT_AVALON_H

// /////////////////////////////////////////////////////////////////
//
// ALT_AVALON_MEMMAP_SLAVE
//
template <int DATA_WIDTH, int DEPTH> class ALT_AVALON_MEMMAP_SLAVE 
: public alt_avalon_slave 
{
	public:
		typedef sc_int<DATA_WIDTH> DATA_TYPE_S;
		typedef sc_uint<DATA_WIDTH> DATA_TYPE_U;
		typedef sc_lv<DATA_WIDTH> DATA_TYPE_LV;
		typedef sc_bv<DATA_WIDTH> DATA_TYPE_BV;
		
	private:
		alt_avalon_bus_matrix &_bus;
		DATA_TYPE_S *_data;
		
        int _effectiveWidthBits;		// NB the avalon bus works with words of size 2^N bytes
        int _effectiveWidthBytes;		// we have to round up the word size to next boundary
        
        int _effectiveBytesOfMemory;
        int _effectiveAddrWidth;
        
        int _wordsOfMemory;
        int _dataWidth;
        int _wordAddrWidth;
        
        DATA_TYPE_S _invalid;

	public:
	
		virtual int getAddressWidth() { return _effectiveAddrWidth; };
		virtual int getDataWidth() { return _effectiveWidthBits; }


		virtual const sc_int_base &acceptBusRead(unsigned int address) 
		{
            unsigned int wordAddress = address/_effectiveWidthBytes;
            if (wordAddress * _effectiveWidthBytes != address ) {
                fprintf(stderr,"Misaligned bus read to address 0x%x (align %d)\n", address, _effectiveWidthBytes);
            }
            if (wordAddress>_wordsOfMemory) {
                fprintf(stderr,"Read outside of slave's area (addr=0x%x,size=0x%x)\n",wordAddress,_wordsOfMemory);
                return _invalid;
            } else {
                //printf("Bus Read %d=word[%d]\n", (int)_data[wordAddress], (int)wordAddress); fflush(stdout);
    			return _data[wordAddress];
            }
		};
			
		virtual void acceptBusWrite(unsigned int address, const sc_int_base &data) 
		{
            unsigned int wordAddress = address/_effectiveWidthBytes;
            if (wordAddress * _effectiveWidthBytes != address ) {
                fprintf(stderr,"Misaligned bus write to address 0x%x (align %d)\n", address, _effectiveWidthBytes);
            }
            if (wordAddress>_wordsOfMemory)
                fprintf(stderr,"Write outside of slave's area (addr=0x%x,size=0x%x)\n",wordAddress,_wordsOfMemory);
            else {
                //printf("Bus Write word[%d]=%d\n", (int)wordAddress, (int)data); fflush(stdout);
    			_data[wordAddress] = data;
            }
		};
		

		const DATA_TYPE_S &readSI(unsigned int wordAddress) 
		{
			_bus.yield();
            //printf("CUSP Read %d=word[%d]\n", (int)_data[wordAddress], (int)wordAddress); fflush(stdout);
			return _data[wordAddress];
		};
		
		void writeSI(unsigned int wordAddress, const DATA_TYPE_S &data)
		{
            //printf("CUSP Write word[%d]=%d\n", (int)wordAddress, (int)data); fflush(stdout);
			_data[wordAddress] = data;
		};

		const DATA_TYPE_U readUI(unsigned int wordAddress) 
		{
			_bus.yield();
            //printf("CUSP Read %d=word[%d]\n", (int)_data[wordAddress], (int)wordAddress); fflush(stdout);
			return (DATA_TYPE_U) _data[wordAddress];
		};
		
		void writeUI(unsigned int wordAddress, const DATA_TYPE_U &data)
		{
            //printf("CUSP Write word[%d]=%s\n", (int)wordAddress, data.to_string().c_str()); fflush(stdout);
			_data[wordAddress] = data;
		};



		const DATA_TYPE_LV readLV(unsigned int wordAddress) 
		{
			_bus.yield();
            //printf("CUSP Read %s=word[%d]\n", _data[wordAddress].to_string().c_str(), (int)wordAddress); fflush(stdout);
			return (DATA_TYPE_LV) _data[wordAddress];
		};
		
		void writeLV(unsigned int wordAddress, const DATA_TYPE_LV &data)
		{
			_data[wordAddress] = data;
            //printf("CUSP Write word[%d]=%s\n", (int)wordAddress, data.to_string().c_str()); fflush(stdout);
		};
		

		const DATA_TYPE_BV readBV(unsigned int wordAddress) 
		{
			_bus.yield();
            //printf("CUSP Read %s=word[%d]\n", _data[wordAddress].to_string().c_str(), (int)wordAddress); fflush(stdout);
			return (DATA_TYPE_BV) _data[wordAddress].range(DATA_WIDTH-1, 0);
		};
		
		void writeBV(unsigned int wordAddress, const DATA_TYPE_BV &data)
		{
            //printf("CUSP Write word[%d]=%s\n", (int)wordAddress, data.to_string().c_str()); fflush(stdout);
			_data[wordAddress].range(DATA_WIDTH-1,0) = data;
		};

		
		ALT_AVALON_MEMMAP_SLAVE(alt_avalon_bus_matrix &bus) 
			: alt_avalon_slave(bus), _bus(bus) {
				
                _invalid = -1;
              
				_wordsOfMemory  = DEPTH;
                _dataWidth = DATA_WIDTH;

				// work out our word size as a 2^N value
                _effectiveWidthBits = 8;
				_effectiveAddrWidth = 0;
                while (_effectiveWidthBits < DATA_WIDTH) {
                	_effectiveAddrWidth += 1;
                	_effectiveWidthBits *= 2;
                }                
                _effectiveWidthBytes = _effectiveWidthBits / 8;
                
                //
                int wordAddressSize = 1;
                int wordsAddressed = 2;
                while (wordsAddressed < DEPTH) {
                	wordAddressSize += 1;
                	wordsAddressed *= 2;
                }                
				_effectiveAddrWidth += wordAddressSize;

				_data = new DATA_TYPE_S[_wordsOfMemory];
		};
			
		virtual ~ALT_AVALON_MEMMAP_SLAVE()  {};
};



// /////////////////////////////////////////////////////////////////
//
// ALT_AVALON_SINK
//
template <int WIDTH=16, int DEPTH=32> class ALT_AVALON_SINK
: public alt_avalon_slave 
{
	public:
		typedef sc_int<WIDTH> DATA_TYPE_S;
		typedef sc_uint<WIDTH> DATA_TYPE_U;
		typedef sc_lv<WIDTH> DATA_TYPE_LV;
		typedef sc_bv<WIDTH> DATA_TYPE_BV;
		
	private:
		alt_avalon_bus_matrix &_bus;
		
        int _effectiveWidthBits;		// NB the avalon bus works with words of size 2^N bytes
        int _effectiveWidthBytes;		// we have to round up the word size to next boundary
        
        
        
        DATA_TYPE_S _fifo[DEPTH];
        unsigned int _rptr;
        unsigned int _wptr;
        unsigned int _used;

        sc_event _rd_event;
        sc_event _wr_event;

        const DATA_TYPE_S _read()
        {
            while (_used == 0) wait(_wr_event);
            return _read_nb();
        }

        const DATA_TYPE_S _read_nb()
        {
            if (_used == 0 ) return DATA_TYPE_S(-1);
            DATA_TYPE_S value = _fifo[_rptr++]; // need to take a copy here otherwise ....
            _rptr %= DEPTH;
            _used--;
            _rd_event.notify();        // .. could be overwritten here
			return value;               
        }


	public:
	
		virtual int getAddressWidth() { return 0; };
		virtual int getDataWidth() { return _effectiveWidthBits; };


        // --------------------------------------------

		virtual const sc_int_base &acceptBusRead(unsigned int address) 
		{
            // cannot read from a sink
            fprintf(stderr,"Attempting to read from an Avalon Sink");
            throw new alt_exception_bus_read_error();
		};
			
		virtual void acceptBusWrite(unsigned int address, const sc_int_base &data) 
		{
            while (_used == DEPTH) wait(_rd_event);
            _fifo[_wptr++] = data;
            _wptr %= DEPTH;
            _used++;
            _wr_event.notify();        
		};

        // --------------------------------------------


		const DATA_TYPE_S  readSI() 	{ return (DATA_TYPE_S)  _read(); 	};
		const DATA_TYPE_U  readUI() 	{ return (DATA_TYPE_U)  _read(); 	};
		const DATA_TYPE_LV readLV() 	{ return (DATA_TYPE_LV) _read(); 	};
		const DATA_TYPE_BV readBV() 	{ return (DATA_TYPE_BV) _read(); 	};
		
        const DATA_TYPE_S  creadSI(bool doRead) 	{ return doRead ? (DATA_TYPE_S)  _read() : DATA_TYPE_S (0);	};
        const DATA_TYPE_U  creadUI(bool doRead) 	{ return doRead ? (DATA_TYPE_U)  _read() : DATA_TYPE_U (0);	};
        const DATA_TYPE_LV creadLV(bool doRead) 	{ return doRead ? (DATA_TYPE_LV) _read() : DATA_TYPE_LV(0); };
        const DATA_TYPE_BV creadBV(bool doRead) 	{ return doRead ? (DATA_TYPE_BV) _read() : DATA_TYPE_BV(0); };

   		const DATA_TYPE_S  nbreadSI() 	{ return (DATA_TYPE_S)  _read_nb();	};
		const DATA_TYPE_U  nbreadUI() 	{ return (DATA_TYPE_U)  _read_nb();	};
		const DATA_TYPE_LV nbreadLV() 	{ return (DATA_TYPE_LV) _read_nb();	};
		const DATA_TYPE_BV nbreadBV() 	{ return (DATA_TYPE_BV) _read_nb();	};
		
        const DATA_TYPE_S  cnbreadSI(bool doRead) 	{ return doRead ? (DATA_TYPE_S)  _read_nb() : DATA_TYPE_S (0); };
        const DATA_TYPE_U  cnbreadUI(bool doRead) 	{ return doRead ? (DATA_TYPE_U)  _read_nb() : DATA_TYPE_U (0); };
        const DATA_TYPE_LV cnbreadLV(bool doRead) 	{ return doRead ? (DATA_TYPE_LV) _read_nb() : DATA_TYPE_LV(0); };
        const DATA_TYPE_BV cnbreadBV(bool doRead) 	{ return doRead ? (DATA_TYPE_BV) _read_nb() : DATA_TYPE_BV(0); };

        const bool isEmpty() { return _used == 0; };

		
		ALT_AVALON_SINK(alt_avalon_bus_matrix &bus) 
			: alt_avalon_slave(bus), _bus(bus), _rptr(0), _wptr(0), _used(0) {
				              

				// work out our word size as a 2^N value
                _effectiveWidthBits = 8;
                while (_effectiveWidthBits < WIDTH) {
                	_effectiveWidthBits *= 2;
                }                
                _effectiveWidthBytes = _effectiveWidthBits / 8;
                
                //
        };
			
		virtual ~ALT_AVALON_SINK()  {};
};



// /////////////////////////////////////////////////////////////////
//
// ALT_AVALON_SOURCE
//
template <int WIDTH=16, int DEPTH=32> class ALT_AVALON_SOURCE
: public alt_avalon_slave 
{

    public:
		typedef sc_int<WIDTH> DATA_TYPE_S;
		typedef sc_uint<WIDTH> DATA_TYPE_U;
		typedef sc_lv<WIDTH> DATA_TYPE_LV;
		typedef sc_bv<WIDTH> DATA_TYPE_BV;
		
	private:
		alt_avalon_bus_matrix &_bus;
		
        int _effectiveWidthBits;		// NB the avalon bus works with words of size 2^N bytes
        int _effectiveWidthBytes;		// we have to round up the word size to next boundary


        DATA_TYPE_S _fifo[DEPTH];
        unsigned int _rptr;
        unsigned int _wptr;
        unsigned int _used;

        sc_event _rd_event;
        sc_event _wr_event;


        void _write(const DATA_TYPE_S &data)
        {
            while (_used == DEPTH) wait(_rd_event);
            _write_nb(data);
        }

        void _write_nb(const DATA_TYPE_S &data)
        {
            if (_used == DEPTH ) return;
            _fifo[_wptr++] = data;
            _wptr %= DEPTH;
            _used++;
            _wr_event.notify();
			return;
        }

    public:
	
		virtual int getAddressWidth() { return 0; };
		virtual int getDataWidth() { return _effectiveWidthBits; }



        // --------------------------------------------

		virtual const sc_int_base &acceptBusRead(unsigned int address) 
		{
            while (_used == 0) wait(_wr_event);
            const DATA_TYPE_S &data = _fifo[_rptr++];
            _rptr %= DEPTH;
            _used--;
			_rd_event.notify();
            return data;
		};
			
		virtual void acceptBusWrite(unsigned int address, const sc_int_base &data) 
		{
            // cannot write to a source
            fprintf(stderr,"Attempting to write to an Avalon Source");
            throw new alt_exception_bus_write_error();
		};



		void writeSI(const DATA_TYPE_S  &data) 	{ _write(data); };
		void writeUI(const DATA_TYPE_U  &data) 	{ _write(DATA_TYPE_S(data)); };
		void writeLV(const DATA_TYPE_LV &data) 	{ _write(DATA_TYPE_S(data)); };
		void writeBV(const DATA_TYPE_BV &data) 	{ _write(DATA_TYPE_S(data)); };
		
        void nbwriteSI(const DATA_TYPE_S  &data) { _write_nb(data); };
        void nbwriteUI(const DATA_TYPE_U  &data) { _write_nb(DATA_TYPE_S(data)); };
        void nbwriteLV(const DATA_TYPE_LV &data) { _write_nb(DATA_TYPE_S(data)); };
        void nbwriteBV(const DATA_TYPE_BV &data) { _write_nb(DATA_TYPE_S(data)); };

		void cwriteSI(const bool doWrite, const DATA_TYPE_S  &data) 	{ if (doWrite) _write(data); };
		void cwriteUI(const bool doWrite, const DATA_TYPE_U  &data) 	{ if (doWrite) _write(DATA_TYPE_S(data)); };
		void cwriteLV(const bool doWrite, const DATA_TYPE_LV &data) 	{ if (doWrite) _write(DATA_TYPE_S(data)); };
		void cwriteBV(const bool doWrite, const DATA_TYPE_BV &data) 	{ if (doWrite) _write(DATA_TYPE_S(data)); };

   		void cnbwriteSI(const bool doWrite, const DATA_TYPE_S  &data) 	{ if (doWrite) _write_nb(data); };
		void cnbwriteUI(const bool doWrite, const DATA_TYPE_U  &data) 	{ if (doWrite) _write_nb(DATA_TYPE_S(data)); };
		void cnbwriteLV(const bool doWrite, const DATA_TYPE_LV &data) 	{ if (doWrite) _write_nb(DATA_TYPE_S(data)); };
		void cnbwriteBV(const bool doWrite, const DATA_TYPE_BV &data) 	{ if (doWrite) _write_nb(DATA_TYPE_S(data)); };


        const bool isFull() { return _used == DEPTH; };



		ALT_AVALON_SOURCE(alt_avalon_bus_matrix &bus) 
			: alt_avalon_slave(bus), _bus(bus), _rptr(0), _wptr(0), _used(0) {
				              

				// work out our word size as a 2^N value
                _effectiveWidthBits = 8;
                while (_effectiveWidthBits < WIDTH) {
                	_effectiveWidthBits *= 2;
                }                
                _effectiveWidthBytes = _effectiveWidthBits / 8;
                
                //
        };
			
		virtual ~ALT_AVALON_SOURCE()  {};
};



// /////////////////////////////////////////////////////////////////
//
// ALT_AVALON_MASTER
//
template <int DATA_WIDTH, int BYTE_ADDR_WIDTH> class ALT_AVALON_MASTER : public alt_avalon_master {
	private:
		alt_avalon_bus_matrix &_bus;
		int _effectiveWidthBits;

	public:
		typedef sc_int<DATA_WIDTH> DATA_TYPE_S;
		typedef sc_uint<DATA_WIDTH> DATA_TYPE_U;
	
		virtual int getAddressWidth() { return BYTE_ADDR_WIDTH; };
		virtual int getDataWidth() { return _effectiveWidthBits; }
		
		DATA_TYPE_S busReadSI(unsigned int address) 
		{
			_bus.yield();
			return (DATA_TYPE_S) _bus.read(*this, address); 
		};
		
		void busWriteSI(unsigned int address, const DATA_TYPE_S data) 
		{ 
			_bus.write(*this, address, data); 
		} ;

        void busByteWriteSI(unsigned int address, sc_uint<(DATA_WIDTH+7)/8> byteEnable, const DATA_TYPE_S data) 
        { 
            DATA_TYPE_S rdata = busReadSI(address);
            if (byteEnable.bit(0)) rdata.range(7,0) = data.range(7,0);
            if (byteEnable.bit(1)) rdata.range(15,8) = data.range(15,8);
            if (byteEnable.bit(2)) rdata.range(23,16) = data.range(23,16);
            if (byteEnable.bit(3)) rdata.range(31,24) = data.range(31,24);
            busWriteSI(address,rdata);
        } ;

		DATA_TYPE_U busReadUI(unsigned int address) 
		{
			_bus.yield();
			return (DATA_TYPE_U) _bus.read(*this, address); 
		};
		
		void busWriteUI(unsigned int address, const DATA_TYPE_U data) 
		{ 
			_bus.write(*this, address, (DATA_TYPE_S)data); 
		} ;
     
        void busByteWriteUI(unsigned int address, sc_uint<(DATA_WIDTH+7)/8> byteEnable, const DATA_TYPE_U data) 
        { 
            DATA_TYPE_U rdata = busReadUI(address);
            if (byteEnable.bit(0)) rdata.range(7,0) = data.range(7,0);
            if (byteEnable.bit(1)) rdata.range(15,8) = data.range(15,8);
            if (byteEnable.bit(2)) rdata.range(23,16) = data.range(23,16);
            if (byteEnable.bit(3)) rdata.range(31,24) = data.range(31,24);
            busWriteUI(address,rdata);
        } ;

		
		ALT_AVALON_MASTER(alt_avalon_bus_matrix &bus) 
			: alt_avalon_master(bus), _bus(bus) {};
			
		virtual ~ALT_AVALON_MASTER()  {
            _effectiveWidthBits = 8;
            while (_effectiveWidthBits < DATA_WIDTH) _effectiveWidthBits *= 2;
        };
};


// /////////////////////////////////////////////////////////////////
//
// ALT_AVALON_MASTER_FIFO
//
template <int DATA_WIDTH, int BYTE_ADDR_WIDTH> class ALT_AVALON_MASTER_FIFO : public alt_avalon_master {
	private:
		alt_avalon_bus_matrix &_bus;
		int _effectiveWidthBits;
        sc_fifo< sc_int<64> > _readFifo;
        

	public:
		typedef sc_int<DATA_WIDTH> DATA_TYPE_S;
		typedef sc_uint<DATA_WIDTH> DATA_TYPE_U;
	
		virtual int getAddressWidth() { return BYTE_ADDR_WIDTH; };
		virtual int getDataWidth() { return _effectiveWidthBits; }
		
		void busPostReadSI(unsigned int address) 
		{
			_bus.yield();
			_readFifo.write(_bus.read(*this,address));
		};
		
		DATA_TYPE_S busCollectReadSI() 
		{
			_bus.yield();
			return (DATA_TYPE_S) _readFifo.read(); 
		};
		
        DATA_TYPE_S busReadSI(unsigned int address) 
        {
            _bus.yield();
            return (DATA_TYPE_S) _bus.read(*this, address); 
        };		
		
		void busWriteSI(unsigned int address, const DATA_TYPE_S data) 
		{ 
			_bus.write(*this, address, data); 
		} ;

        void busByteWriteSI(unsigned int address, sc_uint<(DATA_WIDTH+7)/8> byteEnable, const DATA_TYPE_S data) 
        {         	
            DATA_TYPE_S rdata = busReadSI(address);
            if (byteEnable.bit(0)) rdata.range(7,0) = data.range(7,0);
            if (byteEnable.bit(1)) rdata.range(15,8) = data.range(15,8);
            if (byteEnable.bit(2)) rdata.range(23,16) = data.range(23,16);
            if (byteEnable.bit(3)) rdata.range(31,24) = data.range(31,24);
            busWriteSI(address,rdata);
        } ;

        DATA_TYPE_U busReadUI(unsigned int address) 
        {
            _bus.yield();
            return (DATA_TYPE_U) _bus.read(*this, address); 
        };
 
        void busPostReadUI(unsigned int address) 
		{
			_bus.yield();
			_readFifo.write(_bus.read(*this, address));
		};
		
		DATA_TYPE_U busCollectReadUI() 
		{
			_bus.yield();
			return (DATA_TYPE_U) _readFifo.read(); 
		};
		
				
		void busWriteUI(unsigned int address, const DATA_TYPE_U data) 
		{ 
			_bus.write(*this, address, (DATA_TYPE_S)data); 
		} ;
		
		void busByteWriteUI(unsigned int address, sc_uint<(DATA_WIDTH+7)/8> byteEnable, const DATA_TYPE_U data) 
        { 
            DATA_TYPE_U rdata = busReadUI(address); 
            if (byteEnable.bit(0)) rdata.range(7,0) = data.range(7,0);
            if (byteEnable.bit(1)) rdata.range(15,8) = data.range(15,8);
            if (byteEnable.bit(2)) rdata.range(23,16) = data.range(23,16);
            if (byteEnable.bit(3)) rdata.range(31,24) = data.range(31,24);
            busWriteUI(address,rdata);
        } ;

		
		ALT_AVALON_MASTER_FIFO(alt_avalon_bus_matrix &bus) 
			: alt_avalon_master(bus), _bus(bus) , _readFifo(16){};
			
		virtual ~ALT_AVALON_MASTER_FIFO()  {
            _effectiveWidthBits = 8;
            while (_effectiveWidthBits < DATA_WIDTH) _effectiveWidthBits *= 2;
        };
};



#endif // __ALT_AVALON_H


