/*******************************************************************
 * 
 * Arithemetic unit (AU) function units.
 * 
 * 
 * 
 * Abbreviations :
 *  W          = WIDTH
 *  L          = LATENCY
 *  R          = RESTART
 *  IW         = INTEGER WIDTH (FOR FIXED POINT)
 *  QM         = QUANTIZATION MODE(FOR FIXED POINT)
 *  OFM        = OVERFLOW MODE (FOR FIXED POINT)
 *  NSB        = NUMBER OF SATURATED BITS (USED WITH OFM)
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 *  (2) ALT_AU_INT, ALT_AU_UINT, ALT_AU_FIXED and ALT_AU_UFIXED 
 *      are not currently supported
 * 
 * 
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif


#ifndef __ALT_AU_H
#define __ALT_AU_H
//--------------------------------------------------------------------
// Arithemetic function unit and its named static operators
//--------------------------------------------------------------------

template <int W = 16, int L = 1, int R = 1> 
class ALT_AU  
{

public:
   //--------------------------------------------------------------------
   // Unsigned integer 
   // conditional addition/subtraction/loading/synchronous-clear
   // Conditionals are enable, doSClr, doSLd and doSub in that order 
   // of precedence.
   //--------------------------------------------------------------------
    static sc_uint<W> cAddSubSLdSClrUI(
                                        const sc_uint<W>       &a,
                                        const sc_uint<W>       &b,
                                        const sc_uint<W>       &nextVal, 
                                        const sc_uint<W>       &currVal, 
                                        bool             enable,
                                        bool             doSClr,
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        if(enable)
        {
            if(doSClr)
            {
                return 0;
            }
            else
            {
                if(doSLd)
                {
                    return nextVal;
                }
                else
                {
                    sc_uint<W> result;
                      
                    if(doSub)
                    {
                        result = a - b;
                        return result;
                    }
                    else
                    {
                        result = a + b;
                        return result;
                    }
                }
            }
        }
        else
        {
            return currVal;
        }    
    }
   
   
   //--------------------------------------------------------------------
   // Signed integer 
   // conditional addition/subtraction/loading/synchronous-clear
   // Conditionals are enable, doSClr, doSLd and doSub in that order 
   // of precedence.
   //--------------------------------------------------------------------
   static sc_int<W> cAddSubSLdSClrSI(
                                        const sc_int<W>        &a, 
                                        const sc_int<W>        &b, 
                                        const sc_int<W>        &nextVal, 
                                        const sc_int<W>        &currVal, 
                                        bool             enable,
                                        bool             doSClr,
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        if(enable)
        {
            if(doSClr)
            {
                return 0;
            }
            else
            {
                if(doSLd)
                {
                    return nextVal;
                }
                else
                {
                    sc_int<W> result;
                      
                    if(doSub)
                    {
                        result = a - b;
                        return result;
                    }
                    else
                    {
                        result = a + b;
                        return result;
                    }
                }
            }
        }
        else
        {
            return currVal;
        }    
    }
    
    //--------------------------------------------------------------------
    // Unsigned fix 
    // Conditional addition/subtraction/loading/synchronous-clear
    // Conditionals are enable, doSClr, doSLd and doSub in that order 
    // of precedence.
    //--------------------------------------------------------------------
    static sc_ufix cAddSubSLdSClrUF(
                                  const sc_ufix    &a, 
                                  const sc_ufix    &b, 
                                  const sc_ufix    &nextVal, 
                                  const sc_ufix    &currVal, 
                                  bool       enable,
                                  bool       doSClr,
                                  bool       doSLd,
                                  bool       doSub
                                  )
    {
        return cAddSubSLdSClrUF(a, b, nextVal, currVal, enable, doSClr, doSLd, doSub, 0);
    }
    
    
    //--------------------------------------------------------------------
    // Signed fix 
    // conditional addition/subtraction/loading/synchronous-clear
    // Conditionals are enable, doSClr, doSLd and doSub in that order of
    // of precedence.
    //--------------------------------------------------------------------
    static sc_fix cAddSubSLdSClrSF(
                                  const sc_fix    &a, 
                                  const sc_fix    &b, 
                                  const sc_fix    &nextVal, 
                                  const sc_fix    &currVal, 
                                  bool      enable,
                                  bool      doSClr,
                                  bool      doSLd,
                                  bool      doSub
                                  )
    {
        return cAddSubSLdSClrSF(a, b, nextVal, currVal, enable, doSClr, doSLd, doSub, 0);
    }
    
    //--------------------------------------------------------------------
    // Integer/Fixed conditional loading/synchronous-clear
    //--------------------------------------------------------------------
    static sc_ufix cSLdSClrUF(
                                  const sc_ufix     &nextVal, 
                                  const sc_ufix     &currVal, 
                                  bool             enable,
                                  bool             doSClr,
                                  bool             doSLd
                                  ) 
    {
        return cAddSubSLdSClrUF(0, 0, nextVal, currVal, enable&&(doSClr||doSLd), doSClr, doSLd, false);
    }
    
    static sc_fix cSLdSClrSF(
                                  const sc_fix     &nextVal, 
                                  const sc_fix     &currVal, 
                                  bool             enable,
                                  bool             doSClr,
                                  bool             doSLd
                                  ) 
    {
        return cAddSubSLdSClrSF(0, 0, nextVal, currVal, enable&&(doSClr||doSLd), doSClr, doSLd, false);
    }    
    
    static sc_uint<W> cSLdSClrUI(
                                  const sc_uint<W>     &nextVal, 
                                  const sc_uint<W>     &currVal, 
                                  bool             enable,
                                  bool             doSClr,
                                  bool             doSLd
                                  ) 
    {
        return cAddSubSLdSClrUI(0, 0, nextVal, currVal, enable&&(doSClr||doSLd), doSClr, doSLd, false);
    }                 
    
    static sc_int<W> cSLdSClrSI(
                                  const sc_int<W>     &nextVal, 
                                  const sc_int<W>     &currVal, 
                                  bool             enable,
                                  bool             doSClr,
                                  bool             doSLd
                                  ) 
    {
        return cAddSubSLdSClrSI(0, 0, nextVal, currVal, enable&&(doSClr||doSLd), doSClr, doSLd, false);
    }                       
    //--------------------------------------------------------------------
    // Bit vector conditional loading/synchronous-clear
    //--------------------------------------------------------------------
    static sc_bv<W> cSLdSClrBV(
                                  const sc_bv<W>     &nextVal, 
                                  const sc_bv<W>     &currVal, 
                                  bool             enable,
                                  bool             doSClr,
                                  bool             doSLd
                                  )
    {
        if(enable)
        {
            if(doSClr)
            {
                sc_bv<W> result;
                result = 0;
                return result;
            }
            else
            {
                if(doSLd)
                {
                    return nextVal;
                }
                else
                {
                    return currVal;
                }
            }
        }
        else
        {
            return currVal;
        }    
    }
    
    
    //--------------------------------------------------------------------
    // add() functions
    //--------------------------------------------------------------------
    static sc_uint<W> addUI(
                                const sc_uint<W> &a, 
                                const sc_uint<W> &b
                                )
    {
        sc_uint<W> result = a+b;
        return result;
    }

    static sc_int<W> addSI(
                               const sc_int<W> &a, 
                               const sc_int<W> &b
                               )
    {
        sc_int<W> result = a+b;
        return result;
    }
   
    static sc_ufix addUF(
                         const sc_ufix &a, 
                         const sc_ufix &b
                         )
    {
       	int intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	
        sc_ufix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        sc_ufix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
            
        ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
        
        sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        
        
        result = aAligned + bAligned;
        return result;
    }
    
    static sc_fix addSF(
                       const sc_fix &a, 
                       const sc_fix &b
                       )
    {
       	int intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	
        sc_fix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
            
        ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
        
        sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        
        
        result = aAligned + bAligned;
        return result;
    }
   
    
   
    //--------------------------------------------------------------------
    // sub() functions
    //--------------------------------------------------------------------
    static sc_uint<W> subUI(
                                const sc_uint<W> &a, 
                                const sc_uint<W> &b
                                )
    {
        sc_uint<W> result = a-b;
        return result;
    }

    static sc_int<W> subSI(
                               const sc_int<W> &a, 
                               const sc_int<W> &b
                               )
    {
        sc_int<W> result = a-b;
        return result;
    }
   
    static sc_ufix subUF(
                         const sc_ufix &a, 
                         const sc_ufix &b
                         )
    {
       	int intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	
        sc_ufix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        sc_ufix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
            
        ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
        
        sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        
        
        result = aAligned - bAligned;
        return result;
    }
    
    static sc_fix subSF(
                        const sc_fix &a, 
                        const sc_fix &b
                        )
    {
       	int intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	
        sc_fix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        sc_fix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        aAligned = a;
        bAligned = b;
            
        ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
        ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
        
        sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        
        
        result = aAligned - bAligned;
        return result;
    }
    
    //--------------------------------------------------------------------
    // cAddSubSLd() functions
    //--------------------------------------------------------------------
    static sc_uint<W> cAddSubSLdUI(
                                        const sc_uint<W>       &a, 
                                        const sc_uint<W>       &b, 
                                        const sc_uint<W>       &nextVal, 
                                        const sc_uint<W>       &currVal, 
                                        bool             enable,
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrUI(a, b, nextVal, currVal, enable, false, doSLd, doSub);
    }
   
   
    static sc_int<W> cAddSubSLdSI(
                                        const sc_int<W>        &a, 
                                        const sc_int<W>        &b, 
                                        const sc_int<W>        &nextVal, 
                                        const sc_int<W>        &currVal, 
                                        bool             enable,
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrSI(a, b, nextVal, currVal, enable, false, doSLd, doSub);
    }
    
    
    static sc_ufix cAddSubSLdUF(
                                  const sc_ufix    &a, 
                                  const sc_ufix    &b, 
                                  const sc_ufix    &nextVal, 
                                  const sc_ufix    &currVal, 
                                  bool       enable,
                                  bool       doSLd,
                                  bool       doSub
                                  )
    {
        return cAddSubSLdSClrUF(a, b, nextVal, currVal, enable, false, doSLd, doSub, 0);
    }
    
    
   
    static sc_fix cAddSubSLdSF(
                                  const sc_fix    &a, 
                                  const sc_fix    &b, 
                                  const sc_fix    &nextVal, 
                                  const sc_fix    &currVal, 
                                  bool      enable,
                                  bool      doSLd,
                                  bool      doSub
                                  )
    {
        return cAddSubSLdSClrSF(a, b, nextVal, currVal, enable, false, doSLd, doSub, 0);
    }
    
    
    //--------------------------------------------------------------------
    // addSubSLd() functions
    //--------------------------------------------------------------------
    static sc_uint<W> addSubSLdUI(
                                        const sc_uint<W>       &a, 
                                        const sc_uint<W>       &b, 
                                        const sc_uint<W>       &nextVal, 
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrUI(a, b, nextVal, nextVal, true, false, doSLd, doSub);
    }
   
   
    static sc_int<W> addSubSLdSI(
                                        const sc_int<W>        &a, 
                                        const sc_int<W>        &b, 
                                        const sc_int<W>    &nextVal, 
                                        bool             doSLd,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrSI(a, b, nextVal, nextVal, true, false, doSLd, doSub);
    }
    
    
    static sc_ufix addSubSLdUF(
                                  const sc_ufix    &a, 
                                  const sc_ufix    &b, 
                                  const sc_ufix    &nextVal, 
                                  bool       doSLd,
                                  bool       doSub
                                  )
    {
        return cAddSubSLdSClrUF(a, b, nextVal, nextVal, true, false, doSLd, doSub, 0);
    }
    
    
   
    static sc_fix addSubSLdSF(
                                  const sc_fix    &a, 
                                  const sc_fix    &b, 
                                  const sc_fix    &nextVal, 
                                  bool      doSLd,
                                  bool      doSub
                                  )
    {
        return cAddSubSLdSClrSF(a, b, nextVal, nextVal, true, false, doSLd, doSub, 0);
    }    

    //--------------------------------------------------------------------
    // addSubSLdSClr() functions
    //--------------------------------------------------------------------
    static sc_uint<W> addSubSLdSClrUI(
                                        const sc_uint<W>       &a, 
                                        const sc_uint<W>       &b, 
                                        const sc_uint<W>       &nextVal, 
                                        bool             doSLd,
                                        bool             doSClr,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrUI(a, b, nextVal, nextVal, true, doSClr, doSLd, doSub);
    }
   
   
    static sc_int<W> addSubSLdSClrSI(
                                        const sc_int<W>        &a, 
                                        const sc_int<W>        &b, 
                                        const sc_int<W>    &nextVal, 
                                        bool             doSLd,
                                        bool             doSClr,
                                        bool             doSub
                                        )
    {
        return cAddSubSLdSClrSI(a, b, nextVal, nextVal, true, doSClr, doSLd, doSub);
    }
    
    
    static sc_ufix addSubSLdSClrUF(
                                  const sc_ufix    &a, 
                                  const sc_ufix    &b, 
                                  const sc_ufix    &nextVal, 
                                  bool       doSLd,
                                  bool       doSClr,
                                  bool       doSub
                                  )
    {
        return cAddSubSLdSClrUF(a, b, nextVal, nextVal, true, doSClr, doSLd, doSub, 0);
    }
    
    
   
    static sc_fix addSubSLdSClrSF(
                                  const sc_fix    &a, 
                                  const sc_fix    &b, 
                                  const sc_fix    &nextVal, 
                                  bool          doSLd,
                                  bool          doSClr,
                                  bool          doSub
                                  )
    {
        return cAddSubSLdSClrSF(a, b, nextVal, nextVal, true, doSClr, doSLd, doSub, 0);
    }    

   
    //--------------------------------------------------------------------
    // cAddSub() functions
    //--------------------------------------------------------------------
    static sc_uint<W> cAddSubUI(
                                const sc_uint<W> &a, 
                                const sc_uint<W> &b,
                                const sc_uint<W> &currVal,
                                bool enable,
                                bool doSub
                                )
    {
        return cAddSubSLdSClrUI(a, b, 0, currVal, enable, false, false, doSub);
    }

    static sc_int<W> cAddSubSI(
                               const sc_int<W> &a, 
                               const sc_int<W> &b,
                               const sc_int<W> &currVal,
                               bool enable,
                               bool doSub
                               )
    {
        return cAddSubSLdSClrSI(a, b, 0, currVal, enable, false, false, doSub);
    }
   
    static sc_ufix cAddSubUF(
                         const sc_ufix &a, 
                         const sc_ufix &b,
                         const sc_ufix &currVal,
                         bool enable,
                         bool doSub                         
                         )
    {
        return cAddSubSLdSClrUF(a, b, 0, currVal, enable, false, false, doSub, 0);
    }
    
    static sc_fix cAddSubSF(
                       const sc_fix &a, 
                       const sc_fix &b,
                       const sc_fix &currVal,
                       bool enable,
                       bool doSub
                       )
    {
        return cAddSubSLdSClrSF(a, b, 0, currVal, enable, false, false, doSub, 0);
    }    
    
   //--------------------------------------------------------------------
    // addSub() functions
    //--------------------------------------------------------------------
    static sc_uint<W> addSubUI(
                                const sc_uint<W> &a, 
                                const sc_uint<W> &b,
                                bool doSub
                                )
    {
        if (doSub) return subUI(a,b);
        else return addUI(a,b);
    }

    static sc_int<W> addSubSI(
                               const sc_int<W> &a, 
                               const sc_int<W> &b,
                               bool doSub
                               )
    {
        if (doSub) return subSI(a,b);
        else return addSI(a,b);
    }
   
    static sc_ufix addSubUF(
                         const sc_ufix &a, 
                         const sc_ufix &b,
                         bool doSub                         
                         )
    {
        if (doSub) return subUF(a,b);
        else return addUF(a,b);
    }
    
    static sc_fix addSubSF(
                       const sc_fix &a, 
                       const sc_fix &b,
                       bool doSub
                       )
    {
        if (doSub) return subSF(a,b);
        else return addSF(a,b);
    }        
    
    //--------------------------------------------------------------------
    // cLd() functions (Conditional load)
    //--------------------------------------------------------------------
    static sc_uint<W> cLdUI(const sc_uint<W> &c, const sc_uint<W> &currVal, bool doSLd)
    {
        return cAddSubSLdSClrUI(0, 0, c, currVal, doSLd, false, doSLd, false);
    }

    static sc_int<W> cLdSI(const sc_int<W> &c, const sc_int<W> &currVal, bool doSLd)
    {
        return cAddSubSLdSClrSI(0, 0, c, currVal, doSLd, false, doSLd, false);
    }
    
    static sc_bv<W> cLdBV(const sc_bv<W> &c, const sc_bv<W> &currVal, bool doSLd)
    {
        return cSLdSClrBV(c, currVal, doSLd, false, doSLd);
    }
   
    static sc_ufix cLdUF(const sc_ufix &c, const sc_ufix &currVal, bool doSLd)
    {
    	return cAddSubSLdSClrUF(0, 0, c, currVal, doSLd, false, doSLd, false, 0);
    }
    
    static sc_fix cLdSF(const sc_fix &c, const sc_fix &currVal, bool doSLd)
    {
        return cAddSubSLdSClrSF(0, 0, c, currVal, doSLd, false, doSLd, false,  0);
    }
     
    //--------------------------------------------------------------------
    // sLd() functions (Synchronous load)
    //--------------------------------------------------------------------
    static const sc_uint<W> &sLdUI(const sc_uint<W> &c)
    {
        return c;
    }

    static const sc_int<W> &sLdSI(const sc_int<W> &c)
    {
        return c;
    }
   
    static const sc_bv<W> &sLdBV(const sc_bv<W> &c)
    {
        return c;
    }
    
    static const sc_ufix sLdUF(const sc_ufix &c)
    {
		sc_ufix result(W, c.iwl(), SC_TRN, SC_WRAP, 0); 
		result = c;
		ALT_UTIL::warnIfNotEqualsUF(c, result);
		return result;
    }
    
    static const sc_fix sLdSF(const sc_fix &c)
    {
		sc_fix result(W, c.iwl(), SC_TRN, SC_WRAP, 0);
		result = c;
		ALT_UTIL::warnIfNotEqualsSF(c, result);
		return result;
    }
   
    
    //--------------------------------------------------------------------
    // muxLd() functions (Multiplexed load)
    //--------------------------------------------------------------------
    static const sc_uint<W> &muxLdUI(const sc_uint<W> &a, const sc_uint<W> &b, bool loadSecond)
    {             
        return !loadSecond ? a : b;
    }

    static const sc_int<W> &muxLdSI(const sc_int<W> &a, const sc_int<W> &b, bool loadSecond)
    {
        return !loadSecond ? a : b;
    }
   
    static const sc_bv<W> &muxLdBV(const sc_bv<W> &a, const sc_bv<W> &b, bool loadSecond)
    {
        return !loadSecond ? a : b;
    }
    
    static const sc_ufix muxLdUF(const sc_ufix &a, const sc_ufix &b, bool loadSecond)
    {
    	int iwl = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_ufix result(W, iwl, SC_TRN, SC_WRAP, 0); 
		result = !loadSecond ? a : b;
		!loadSecond ? ALT_UTIL::warnIfNotEqualsUF(a, result) : ALT_UTIL::warnIfNotEqualsUF(b, result);
		return result;
    }
    
    static const sc_fix muxLdSF(const sc_fix &a, const sc_fix &b, bool loadSecond)
    {
		int iwl = a.iwl() > b.iwl() ? a.iwl() : b.iwl();
		sc_fix result(W, iwl, SC_TRN, SC_WRAP, 0); 
		result = !loadSecond ? a : b;
		!loadSecond ? ALT_UTIL::warnIfNotEqualsSF(a, result) : ALT_UTIL::warnIfNotEqualsSF(b, result);
		return result;
    }
    
    
    //--------------------------------------------------------------------
    // mCLd() functions (Multiple conditional load)
    //--------------------------------------------------------------------
    static const sc_uint<W> &mCLdUI(const sc_uint<W> &a, const sc_uint<W> &b, sc_uint<W> &currVal, bool loadA, bool loadB)
    {
    	if(loadA) return a;
    	else if(loadB) return b;
    	else return currVal;
    }

    static const sc_int<W> &mCLdSI(const sc_int<W> &a, const sc_int<W> &b, sc_int<W> &currVal, bool loadA, bool loadB)
    {
        if(loadA) return a;
    	else if(loadB) return b;
    	else return currVal;
    }
   
    static const sc_bv<W> &mCLdBV(const sc_bv<W> &a, const sc_bv<W> &b, sc_bv<W> &currVal, bool loadA, bool loadB)
    {
        if(loadA) return a;
    	else if(loadB) return b;
    	else return currVal;
    }
    
    static const sc_ufix mCLdUF(const sc_ufix &a, const sc_ufix &b, sc_ufix &currVal, bool loadA, bool loadB)
    {
		sc_ufix result(W, currVal.iwl(), SC_TRN, SC_WRAP, 0); 
		if(loadA)
		{
			result = a;
			ALT_UTIL::warnIfNotEqualsUF(a, result);
			return result;
		}
		else if(loadB)
		{
			result = b;
			ALT_UTIL::warnIfNotEqualsUF(b, result);
			return result;
		}
		else return currVal;
		
    }
    
    static const sc_fix mCLdSF(const sc_fix &a, const sc_fix &b, sc_fix &currVal, bool loadA, bool loadB)
    {
		sc_fix result(W, currVal.iwl(), SC_TRN, SC_WRAP, 0); 
		if(loadA)
		{
			result = a;
			ALT_UTIL::warnIfNotEqualsSF(a, result);
			return result;
		}
		else if(loadB)
		{
			result = b;
			ALT_UTIL::warnIfNotEqualsSF(b, result);
			return result;
		}
		else return currVal;
    }
    
    
    //--------------------------------------------------------------------
    // sClr() functions (Synchronous clear)
    //--------------------------------------------------------------------
    static sc_uint<W> sClrUI()
    {
        return cAddSubSLdSClrUI(0, 0, 0, 0, true, true, false, false);
    }
    
    static sc_int<W> sClrSI()
    {
        return cAddSubSLdSClrSI(0, 0, 0, 0, true, true, false, false);
    }
   
    static sc_bv<W> sClrBV()
    {
        return cSLdSClrBV(0, 0, true, true, false);
    }
    
    
private:


    //--------------------------------------------------------------------
    // Unsigned fix 
    // conditional addition/subtraction/loading/synchronous-clear
    // Conditionals are enable, doSClr, doSLd and doSub in that order of
    // of precedence.
    // If the integer word length of "a" and "b" don't match, they will be
    // aligned by deleting the LSBs if necessary.
    // If the word length of either "a" or "b" is greater than the input
    // bit width of the fuction unit "W", the LSBs will be deleted.
    //--------------------------------------------------------------------
    static sc_ufix cAddSubSLdSClrUF(
                                  const sc_ufix   &a, 
                                  const sc_ufix   &b, 
                                  const sc_ufix   &nextVal, 
                                  const sc_ufix   &currVal, 
                                  bool       enable,
                                  bool       doSClr,
                                  bool       doSLd,
                                  bool       doSub,
                                  short      iwlMode
                                  )
    {  	
    	
		int intWidthOfOut; 
    	
    	switch(iwlMode)
    	{
    		case 0 :  intWidthOfOut = currVal.iwl();
    				  break;
    		case 1 :  intWidthOfOut = nextVal.iwl();
    				  break;
    		default : intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	}
    	
        if(enable)
        {
        	if(doSClr)
        	{
        		sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        		result = 0;
        		return result;
        	}
        	
            if(doSLd)
            {
			    sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
				result = nextVal;
				ALT_UTIL::warnIfNotEqualsUF(nextVal, result);
				return result;
            }
            else
            {
            	sc_ufix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                sc_ufix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                aAligned = a;
                bAligned = b;
                
                ALT_UTIL::warnIfNotEqualsUF(a, aAligned);
                ALT_UTIL::warnIfNotEqualsUF(b, bAligned);
                
                sc_ufix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                    
                if(doSub)
                {
                    result = aAligned - bAligned;
                    return result;
                }
                else
                {
                    result = aAligned + bAligned;
                    return result;
                }
            }
        }
        else
        {
        	sc_ufix result(W ,intWidthOfOut, SC_TRN, SC_WRAP, 0);
        	result = currVal;
        	ALT_UTIL::warnIfNotEqualsUF(currVal, result);
            return result;
        }    
    }
    
    
    //--------------------------------------------------------------------
    // Signed fix 
    // conditional addition/subtraction/loading/synchronous-clear
    // Conditionals are enable, doSClr, doSLd and doSub in that order of
    // of precedence.
    // If the integer word length of "a" and "b" don't match, they will be
    // aligned by deleting the LSBs if necessary.
    // If the word length of either "a" or "b" is greater than the input
    // bit width of the fuction unit "W", the LSBs will be deleted.
    //--------------------------------------------------------------------
    static sc_fix cAddSubSLdSClrSF(
                                  const sc_fix   &a, 
                                  const sc_fix   &b, 
                                  const sc_fix   &nextVal, 
                                  const sc_fix   &currVal, 
                                  bool      enable,
                                  bool      doSClr,
                                  bool      doSLd,
                                  bool      doSub,
                                  short     iwlMode
                                  )
    {
    	int intWidthOfOut; 
    	
    	switch(iwlMode)
    	{
    		case 0 :  intWidthOfOut = currVal.iwl();
    				  break;
    		case 1 :  intWidthOfOut = nextVal.iwl();
    				  break;
    		default : intWidthOfOut = (a.iwl() > b.iwl() ? a.iwl() : b.iwl());
    	}
    	
        if(enable)
        {
        	if(doSClr)
        	{
        		sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
        		result = 0;
        		return result;
        	}
        	
            if(doSLd)
            {
		        sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
				result = nextVal;
				ALT_UTIL::warnIfNotEqualsSF(nextVal, result);
				return result;
            }
            else
            {
            	sc_fix aAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                sc_fix bAligned(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                aAligned = a;
                bAligned = b;
                    
                ALT_UTIL::warnIfNotEqualsSF(a, aAligned);
                ALT_UTIL::warnIfNotEqualsSF(b, bAligned);
                
                sc_fix result(W, intWidthOfOut, SC_TRN, SC_WRAP, 0);
                
                
                
                if(doSub)
                {
                    result = aAligned - bAligned;
                    return result;
                }
                else
                {
                    result = aAligned + bAligned;
                    return result;
                }
            }
        }
        else
        {
        	sc_fix result(W ,intWidthOfOut, SC_TRN, SC_WRAP, 0);
        	result = currVal;
        	ALT_UTIL::warnIfNotEqualsSF(currVal, result);
            return result;
        }     
    }
};

 /*******************************************************************
 * A number of convenience classes which declare a function unit and
 * declare and bind a variable to it are also provided.
 * 
 * These should generally not be used.
 * 
 * ALT_AU_UINT supports:
 * (1) All of ALT_AU operations for simulation and synthesis
 * (2) All of sc_uint operations for simulation
 * (3) The following sc_uint operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         +  -
 *      Assignment         =  +=  -=  
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_AU_INT supports:
 * (1) All of ALT_AU operations
 * (2) All of sc_int operations (Not all operations are synthesisable)
 * (3) The following sc_int operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  << (as wire or combinational operations)
 *      Arithmetic         +  -
 *      Assignment         =  +=  -=  
 *      Bit Select        []
 *      Part Select       range()
 *      Concatenation     (,)
 * 
 * 
 * ALT_AU_UFIXED supports:
 * (1) All of ALT_AU operations
 * (2) All of sc_ufixed operations for simulation
 * (3) The following sc_ufixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         +  -    
 *      Assignment         =  +=  -=     
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 * ALT_AU_FIXED supports:
 * (1) All of ALT_AU operations
 * (2) All of sc_fixed operations for simulation
 * (3) The following sc_fixed operations for synthesis
 *      Bitwise            ~  &  |  ^    (as combinational operations)
 *      Arithmetic         +  -    
 *      Assignment         =  +=  -=     
 *      Bit Select        []
 *      Part Select       range()
 * (4) All quantization modes for simulation
 * (5) The following quantization modes for synthesis
 *      SC_TRN
 * (6) All overflow modes for simulation
 * (7) The following overflow modes for synthesis
 *      SC_WRAP, NSB = 0;
 * 
 * 
 * ALT_AU_BV supports:
 * (1) All of ALT_AU operations
 * (2) All of sc_bv operations for simulation
 * (3) The following sc_bv operations for synthesis
 *      Bitwise            ~  &  |  ^  >>  <<     (as wire or combinational operations)
 *      Assignment         =  +=  -=
 *      Bit Select        []
 *      Part Select       range()    
 *      Concatenation     (,)
 *      Reduction         and_reduce()  or_reduce()  xor_reduce()(as combinational operations)
 *
 * 
 *
 *  
 *******************************************************************/
//--------------------------------------------------------------------
// Arithemetic function unit with an sc_int variable
//--------------------------------------------------------------------
 
template <int W=16, int L=1, int R=1> 
class ALT_AU_INT : public sc_int<W + 1>, public ALT_AU<W, L, R>   
{
    #define CUSP_NAME ALT_AU_INT    
    #define CUSP_TYPE_NAME ALT_AU_INT<W, L, R>
    #define CUSP_WRAP_SC_INT
    #define CUSP_WRAP_INT_SIZE (W+1)
    #include "cusp/alt_wrap.h"

    CUSP_NAME ( const CUSP_TYPE_NAME& a ) : sc_int<W + 1>( a ) {}

    #undef CUSP_WRAP_INT_SIZE
    #undef CUSP_WRAP_SC_INT
    #undef CUSP_TYPE_NAME
    #undef CUSP_NAME
};


//--------------------------------------------------------------------
// Arithemetic function unit with an sc_uint variable
//--------------------------------------------------------------------

template <int W=16, int L=1, int R=1> 
class ALT_AU_UINT : public sc_uint<W + 1>, public ALT_AU<W, L, R>   
{
    #define CUSP_NAME ALT_AU_UINT
    #define CUSP_TYPE_NAME ALT_AU_UINT<W, L, R>
    #define CUSP_WRAP_SC_UINT
    #define CUSP_WRAP_INT_SIZE (W+1)
    #include "cusp/alt_wrap.h"

    CUSP_NAME ( const CUSP_TYPE_NAME & a ) : sc_uint<W + 1>( a ) {}

    #undef CUSP_WRAP_INT_SIZE
    #undef CUSP_WRAP_SC_UINT
    #undef CUSP_TYPE_NAME
    #undef CUSP_NAME
};


//--------------------------------------------------------------------
// Arithemetic function unit with an sc_fixed variable
//--------------------------------------------------------------------
 
template <int W=16, int L=1, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0>
class ALT_AU_FIXED : public sc_fixed<W+1, IW+1, QM, OFM, NSB+1>, 
                     public ALT_AU<W, L, R>   
{
    #define CUSP_NAME ALT_AU_FIXED
    #define CUSP_TYPE_NAME ALT_AU_FIXED<W, L, R, IW, QM, OFM, NSB>    
    #define CUSP_WRAP_SC_FIXED
    #include "cusp/alt_wrap.h"

    CUSP_NAME ( const CUSP_TYPE_NAME & a ) : sc_fixed<W+1, IW+1, QM, OFM, NSB+1>( a ) {}

    #undef CUSP_WRAP_SC_FIXED
    #undef CUSP_TYPE_NAME
    #undef CUSP_NAME
};
 
 
//--------------------------------------------------------------------
// Arithemetic function unit with an sc_ufixed variable
//--------------------------------------------------------------------

template <int W=16, int L=1, int R=1,
          int IW = 16, sc_q_mode QM = SC_TRN, sc_o_mode OFM = SC_WRAP, int NSB = 0>
class ALT_AU_UFIXED : public sc_ufixed<W+1, IW+1, QM, OFM, NSB+1>, 
                      public ALT_AU<W, L, R>   
{
    #define CUSP_NAME ALT_AU_UFIXED
    #define CUSP_TYPE_NAME ALT_AU_UFIXED<W, L, R, IW, QM, OFM, NSB>
    #define CUSP_WRAP_SC_UFIXED
    #include "cusp/alt_wrap.h"

    CUSP_NAME ( const CUSP_TYPE_NAME & a ) : sc_ufixed<W+1, IW+1, QM, OFM, NSB+1>( a ) {}

    #undef CUSP_WRAP_SC_UFIXED
    #undef CUSP_TYPE_NAME
    #undef CUSP_NAME
};
#endif




