/*******************************************************************
 * 
 * Atlantic Channel (Source/Sink) function units.
 * 
 * Atlantic channels are characterised as as either a SOURCE (producer)
 * or a SINK (consumer of data). Data passes across Atlantic channels in
 * a series of packets. Packets may range in size from 1 word to N words.
 * 
 * By dealing with data in terms packets (or "pages") rather than individual 
 * items more efficient Cusp processors can be produced.
 * 
 * Abbreviations :
 *  T          = A TYPE (e.g. short, char, sc_int<6>, etc)
 * 
 * 
 * ALT_ATLANTIC_SINK<T, PAGE_SIZE, NUM_PAGES >  
 *   - An atlantic sink tha handles data of type T.
 * 
 *   ->getFullPage()
 * 			    wait for an incoming packet to become available
 * 
 *   ->readSinkPage(int offset)
 * 				read word from packet at `offset`
 *    
 *   ->writeSinkPage(int offste, T data)
 * 				write word to packet at `offset`
 * 
 *   ->releasePage()
 *              indicate that the current packet has been processed
 * 
 *   ->getNextPage()
 *              indicate that the current packet has been processed, wait for 
 * 				the next incoming packet to become available
 *
 *   ->read() 
 *              get a full page, read entry at address 0, and return page empty
 * 
 * The Atlantic sink uses a memory to store incoming packets ("pages"). The
 * memory may contain 1 or more pages (specified by NUM_PAGES). This allows
 * packets to be buffered. The default of 2 pages allows one packet to be
 * received while the current packet is being processed.
 * 
 * The first call to the function unit must be getFullPage() to receive
 * a packet full of data.
 * 
 * As packets are stored in a memory they may be accessed in a random way
 * using the read/writeSinkPage functions.
 * 
 * Once a packet has been fully processed releasePage() should be called.
 * 
 * As a convenience getNextPage() releases the current page and invokes
 * getFullPage() to wait for a new packet.
 * 
 * NB The behaviour of the function unit is undefined if the code attempts to 
 * hold multiple pages or to read and write a page if one is not being held.
 * 
 * 
 * 
 * ALT_ATLANTIC_SOURCE<T, PAGE_SIZE, NUM_PAGES >  
 *   - An atlantic source tha handles data of type T.
 * 
 *   ->getEmptyPage()
 * 			    wait for an empty output packet buffer to become available
 * 
 *   ->writeSourcePage(int offset, T data)
 * 				write word to packet at `offset`
 * 
 *   ->readSourcePage(int offset)
 * 				read word from packet at `offset`
 *    
 *   ->releasePage()
 *              indicate that the current packet has been processed
 * 
 *   ->getNextPage()
 *              indicate that the current packet has been processed, wait for 
 * 				the next incoming packet to become available
 *
 *   ->write(T data) 
 *              get an empty page, set entry at address 0 to data, and return full page
 *
 * 
 * The Atlantic source uses a memory to store outgoing packets ("pages"). The
 * memory may contain 1 or more pages (specified by NUM_PAGES). This allows
 * packets to be buffered. The default of 2 pages allows one packet to be
 * transmitted while the next packet is being constructed.
 * 
 * The first call to the function unit must be getEmptyPage() to allocate an
 * empty page buffer.
 * 
 * As packets are stored in a memory they may be accessed in a random way
 * using the read/writeSourcePage functions.
 * 
 * Once a packet has been filled sendPage() should be called.
 * 
 * As a convenience getNextEmptyPage() sends the current page and invokes
 * getEmptyPage() to allocate an empty page.
 * 
 * NB The behaviour of the function unit is undefined if the code attempts to 
 * hold multiple pages or to read and write a page if one is not being held.
 * 
 * 
 * Notes :
 *  (1) Do not include this file directly - include alt_cusp.h
 * 
 * 
 * 
 *******************************************************************/


#ifndef __ALT_CUSP_H
#include <alt_cusp.h>
#endif


#ifndef __ALT_ATLANTIC_H
#define __ALT_ATLANTIC_H

template <class T> class ALT_ATLANTIC_SINK
: virtual public sc_interface
{
        unsigned int page_size;
        T * page;

	public:
		virtual ~ALT_ATLANTIC_SINK() {};
        virtual T read() = 0;
        virtual bool hasDataAvail(void) = 0;
		
		ALT_ATLANTIC_SINK()
		{
			page_size = 1;
			page = NULL;
		}

		void getFullPage()
		{
			for (int i = 0 ; i < page_size ; i++)
				page[i] = read();
		}
		void releasePage() {}        
		const T &readSinkPage(const unsigned int address) { return page[address]; }
        void writeSinkPage(const unsigned int address, const T &data) { page[address] = data; }
		T &operator[](const unsigned int address) { return page[address]; }
        
        void cGetFullPage(bool cond) {
            if (cond) getFullPage();
        }
        void cReleasePage(bool cond) {
            if (cond) releasePage();
        }

        T cRead(bool cond) {
        	static T undefined;
            if (cond) return read();
            return undefined;
        }

		bool getEndPacket(void) { return false; }

		// Setup functions - these must only be called from the constructor
		void setPageSize(int n) { delete page; page = new T[n]; page_size = n; }

		void setBuffers(int n) {} // No effect on behavioural code
		void setSymbolsPerBeat(int spb) {} // No effect on behavioural code
};

template <class T>  class ALT_ATLANTIC_SOURCE
: virtual public sc_interface 
{
        unsigned int page_size;
        T * page;

	public:
		virtual ~ALT_ATLANTIC_SOURCE() {};
        virtual void write(const T &data) = 0;
        virtual bool hasSpaceAvail(void) = 0;

		ALT_ATLANTIC_SOURCE()
		{
			page_size = 1;
			page = NULL;
		}

		virtual void getEmptyPage() {}
		virtual void sendPage()
		{
			for (int i = 0 ; i < page_size ; i++)
				write(page[i]);
		}
        void writeSourcePage(const unsigned int address, const T &data) { page[address] = data; }
		const T &readSourcePage(const unsigned int address) { return page[address]; }
		T &operator[](const unsigned int address) { return page[address]; }

        virtual void cGetEmptyPage(bool cond) { 
            if (cond) getEmptyPage(); 
        }
        virtual void cSendPage(bool cond) {
            if (cond) sendPage();
        }

        virtual void cWrite(const T &data, bool cond) {
            if (cond) write(data);
        }

		void setEndPacket(bool eop) {}
		
		// Setup functions - these must only be called from the constructor
		void setPageSize(int n) { delete page; page = new T[n]; page_size = n; }

		void setBuffers(int n) {} // No effect on behavioural code
		void setSymbolsPerBeat(int spb) {} // No effect on behavioural code
};

#endif
