library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;

entity custom_jtag_counter is
	generic 
		(
			WIDTH		: natural := 22
		);
	port
		(
			clk		: in STD_LOGIC;
			s_load		: in STD_LOGIC;
			s_in		: in STD_LOGIC;
			count		: in STD_LOGIC;
			
			s_out		: out STD_LOGIC;
			p_out		: out STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0)
		);
end entity custom_jtag_counter;

architecture rtl of custom_jtag_counter is	
	signal data_reg		: STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0);
begin
	process(clk, s_load,  s_in, count)
	begin
		if ( rising_edge(clk) ) then
			if ( s_load = '1')  then
				data_reg <= s_in & data_reg(WIDTH - 1 DOWNTO 1);
			elsif (count = '1') then
				data_reg <= data_reg + conv_std_logic_vector(1, WIDTH);
			end if;
		end if;
	end process;
	p_out <= data_reg;
	s_out <= data_reg(0);
end architecture rtl;

package CONSTANTS is
 	constant ALTERA_MFG_ID        : natural := 110;
 	constant PFL_NODE_ID          : natural := 5;
 	constant PFL_VERSION          : natural := 2;
 	constant PFL_INSTANCE         : natural := 0;
 	constant N_NODE_VERSION_BITS  : natural := 5;
 	constant N_NODE_ID_BITS       : natural := 8;
 	constant N_MFG_ID_BITS        : natural := 11;
 	constant N_INSTANCE_BITS      : natural := 8;
 	constant N_SLD_NODE_INFO_BITS : natural := N_NODE_VERSION_BITS + N_NODE_ID_BITS + N_MFG_ID_BITS + N_INSTANCE_BITS;
 	constant N_INFO_BITS          : natural := 8;
 	constant INFO_SHIFT_INST      : natural := 0;
 	constant EN_CONFIG_INST       : natural := 4;
 	constant ADDR_SHIFT_INST      : natural := 2;
 	constant WRITE_INST           : natural := 3;
 	constant READ_INST            : natural := 1;
end package CONSTANTS;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;
use WORK.CONSTANTS.all;
package FUNCTIONS is
	function make_node_info
	(
		NODE_VERSION : natural ;
		NODE_ID      : natural ;
		MFG_ID       : natural ;
		INSTANCE     : natural
	)
	return natural;
end package FUNCTIONS;

package body FUNCTIONS is
	function make_node_info
	(
		NODE_VERSION : natural ;
		NODE_ID      : natural ;
		MFG_ID       : natural ;
		INSTANCE     : natural
	)
	return natural is
		variable info_val : natural;
	begin
		info_val := NODE_VERSION;
		info_val := (info_val * (2 ** N_NODE_ID_BITS)) + NODE_ID;
		info_val := (info_val * (2 ** N_MFG_ID_BITS)) + MFG_ID;
		info_val := (info_val * (2 ** N_INSTANCE_BITS)) + INSTANCE;
		return info_val;
	end make_node_info;
end package body FUNCTIONS;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;
use WORK.CONSTANTS.all;
use WORK.FUNCTIONS.all;

entity alt_pfl_pgm is
	generic 
	(
		PFL_IR_BITS		: natural := 5;
		ADDR_WIDTH		: natural;	-- from 19 for 8Mbits, up to 25 for 512Mbits
		FLASH_DATA_WIDTH: natural := 16;	-- valid values are 8 and 16
		SLD_NODE_INFO	: natural := make_node_info(PFL_VERSION, PFL_NODE_ID, ALTERA_MFG_ID, PFL_INSTANCE)
	);
	port
	(
		pfl_clk			: in STD_LOGIC;
		pfl_nreset		: in STD_LOGIC;
		pfl_flash_access_granted: in STD_LOGIC;
		pfl_flash_access_request: out STD_LOGIC;

--		from HUB	
		ir_in			: in STD_LOGIC_VECTOR (PFL_IR_BITS-1 downto 0);
		ir_out			: out STD_LOGIC_VECTOR (PFL_IR_BITS-1 downto 0);
		tdi				: in STD_LOGIC;
		raw_tck			: in STD_LOGIC;
		tdo				: out STD_LOGIC;
		usr1			: in std_logic;
		jtag_state_sdr	: in STD_LOGIC;
		jtag_state_udr	: in STD_LOGIC;
		jtag_state_e1dr	: in STD_LOGIC;
		jtag_state_rti	: in STD_LOGIC;

--      to CFG
        enable_configuration : out STD_LOGIC;
        enable_nconfig : out STD_LOGIC;
        
-- 		flash output
		flash_addr		: out STD_LOGIC_VECTOR (ADDR_WIDTH-1 downto 0);
		flash_data_in	: in STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
		flash_data_out	: out STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
		flash_read		: out STD_LOGIC;
		flash_write		: out STD_LOGIC;
		flash_select	: out STD_LOGIC
	    );
end entity alt_pfl_pgm;

architecture rtl of alt_pfl_pgm is
	function log2(A: integer) return integer is
	begin
	  for I in 1 to 30 loop  -- Works for up to 32 bit integers
	    if(2**I > A) then return(I-1);  end if;
	  end loop;
	  return(30);
	end;

	--system level
	signal data_inst_dr_scan	: STD_LOGIC;
	signal jtag_rti				: STD_LOGIC;

	--param reg
	signal param1_serial_out	: STD_LOGIC;
	signal param1_data_in		: STD_LOGIC_VECTOR(3 DOWNTO 0);

	signal param2_serial_out	: STD_LOGIC;
	signal param2_data_in		: STD_LOGIC_VECTOR(1 DOWNTO 0);
	signal param2_data_out		: STD_LOGIC_VECTOR(1 DOWNTO 0);
--	signal flash_req_sig		: STD_LOGIC;

	-- jtag flash data reg
	signal jtag_flash_data_in	: STD_LOGIC_VECTOR(FLASH_DATA_WIDTH DOWNTO 0);
	signal jtag_flash_data_out	: STD_LOGIC_VECTOR(FLASH_DATA_WIDTH DOWNTO 0);
	signal jtag_load_flash_data	: STD_LOGIC;
	signal jtag_flash_serial_out: STD_LOGIC;

	--jtag addr_counter
	signal jtag_addr_serial_out	: STD_LOGIC;
	
	-- HUB / instruction
	signal dr_scan				: STD_LOGIC;
	signal data_inst_jtag_udr	: STD_LOGIC;
	signal jtag_sdr				: STD_LOGIC;
   signal jtag_data_inst		: STD_LOGIC;
	signal jtag_addr_inst		: STD_LOGIC;
	signal jtag_info_inst		: STD_LOGIC;

	signal jtag_flash_write		: STD_LOGIC;
	signal jtag_flash_read		: STD_LOGIC;
    
	signal enable_configuration_sig : STD_LOGIC;

	COMPONENT custom_jtag_counter
		generic 
			(
				WIDTH		: natural
			);
		port
			(
				clk			: in STD_LOGIC;
				s_load		: in STD_LOGIC;
				s_in		: in STD_LOGIC;
				count		: in STD_LOGIC;
	
				s_out		: out STD_LOGIC;
				p_out		: out STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0)
			);
	END COMPONENT custom_jtag_counter;	
	COMPONENT lpm_shiftreg
		GENERIC 
		(
			LPM_WIDTH		: POSITIVE;
			LPM_AVALUE		: STRING := "UNUSED";
--			LPM_SVALUE		: NATURAL := 0;
			LPM_PVALUE		: STRING := "UNUSED";
			LPM_DIRECTION	: STRING := "UNUSED";
			LPM_TYPE		: STRING := "LPM_SHIFTREG";
			LPM_HINT		: STRING := "UNUSED"
		);
		PORT 
		(
			data			: IN STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0) := (OTHERS => '0');
			clock			: IN STD_LOGIC;
			enable, shiftin	: IN STD_LOGIC := '1';
			load, sclr, sset, aclr, aset: IN STD_LOGIC := '0';
			q				: OUT STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0);
			shiftout		: OUT STD_LOGIC
		);
	END COMPONENT lpm_shiftreg;

begin
	dr_scan 	   <= not usr1;
	jtag_sdr  	   <= jtag_state_sdr and dr_scan;
	jtag_data_inst	   <= jtag_flash_read or jtag_flash_write;
	data_inst_jtag_udr <= jtag_state_udr and data_inst_dr_scan;
	data_inst_dr_scan  <= dr_scan and jtag_data_inst;
	jtag_rti 	   <= jtag_state_rti;

	enable_configuration_sig <= not ir_in(EN_CONFIG_INST);
	enable_configuration <= enable_configuration_sig;
	jtag_addr_inst	<= ir_in(ADDR_SHIFT_INST);	
	jtag_info_inst 	<= ir_in(INFO_SHIFT_INST);
	jtag_flash_read	<= ir_in(READ_INST);
	jtag_flash_write<= ir_in(write_INST);

    	jtag_addr: custom_jtag_counter
		generic map
		(
			WIDTH 	=> ADDR_WIDTH
		)
		port map
		(
			clk		=> raw_tck,
			s_load	=> jtag_addr_inst and jtag_sdr,
			s_in	=> tdi,
			count	=> data_inst_jtag_udr and not jtag_flash_data_out(0),

			p_out	=> flash_addr,
			s_out	=> jtag_addr_serial_out
		);

	--3 bits for size 1 bit for access granted
	-- parallel load the data when jtag_info_inst is shifted and JSM is not
	-- in SDR..but enabled only in SDR or RTI ..so no risk to assign 
	param1_shiftreg: lpm_shiftreg
		generic map
		(
			LPM_WIDTH 		=> 4,
			LPM_DIRECTION	=> "RIGHT"
		)
		port map
		(
			data			=> param1_data_in,
			clock			=> raw_tck,
			shiftin			=> tdi,
			shiftout		=> param1_serial_out,
			load			=> not jtag_sdr
		);

	param1_data_in(0) <= pfl_flash_access_granted;
	param1_data_in(3 downto 1) <= CONV_STD_LOGIC_VECTOR(ADDR_WIDTH+log2(FLASH_DATA_WIDTH)-23, 3);	-- CFI size (0 for CFI_008, 6 for CFI_512)
	
	-- enable this when no info inst to load the default value..since
	-- on powerup init_conf and pfl_flash_access_request should driven to default value
	-- serial shift the data in SDR and jtag_info_inst
	--1 bit for pfl_flash_access_request 1 bit for ninitconf
	param2_shiftreg: lpm_shiftreg
		generic map
		(
			LPM_WIDTH 		=> 2,
			LPM_DIRECTION	=> "RIGHT"
		)
		port map
		(
			data		=> param2_data_in,
			clock		=> raw_tck,
			enable		=> jtag_info_inst and jtag_sdr,
			shiftin		=> param1_serial_out,
			shiftout	=> param2_serial_out,
			load		=> not jtag_sdr,
			q			=> param2_data_out
		);
	--by default keep this high, software driver will shift 0 when required
	param2_data_in(0) <= '0';
	param2_data_in(1) <= '0';

	--set pfl_flash_access_request output
	pfl_flash_access_request <= not enable_configuration_sig;

	-- in SDR this should be in serial mode, rest of the time in parallel mode
	-- or config controller shifting the data to FPGA
	jtag_load_flash_data <=	(jtag_flash_read and jtag_rti);

	jtag_flash_data_in(FLASH_DATA_WIDTH downto 1) <= flash_data_in(FLASH_DATA_WIDTH-1 downto 0);
	jtag_flash_data_in(0) <= '0';

	jtag_flash_datareg: lpm_shiftreg
		generic map
		(
			LPM_WIDTH 		=> FLASH_DATA_WIDTH + 1,
			LPM_DIRECTION	=> "RIGHT"
		)
		port map
		(
			data			=> jtag_flash_data_in,
			shiftin			=> tdi,
			clock			=> raw_tck,
			enable			=> jtag_load_flash_data or jtag_sdr,
			shiftout		=> jtag_flash_serial_out,
			load			=> jtag_load_flash_data,
			q				=> jtag_flash_data_out
		);

	-- drive out only during flash programing
	drive_flash_bus : process(jtag_flash_data_out, jtag_flash_write, dr_scan)
	begin
			if(jtag_flash_write = '1' and dr_scan = '1') then
				flash_data_out <= jtag_flash_data_out(FLASH_DATA_WIDTH downto 1);
			else
				flash_data_out <= (others => 'Z');
			end if;					
	end process;

	flash_write <= jtag_state_e1dr and jtag_flash_write and dr_scan;
	flash_read <= jtag_flash_read and jtag_rti;
	flash_select <= (jtag_state_e1dr and jtag_flash_write and dr_scan) or (jtag_flash_read and jtag_rti);

	-- tdo mux between the shift register or param register
	tdo_mux :process (param2_serial_out, jtag_flash_serial_out, jtag_addr_inst, jtag_addr_serial_out, jtag_info_inst)
	begin
		if (jtag_info_inst = '1') then
			tdo <= param2_serial_out;
		elsif (jtag_addr_inst = '1') then
			tdo <= jtag_addr_serial_out;
		else
			tdo <= jtag_flash_serial_out;
		end if;
	end process tdo_mux;

	--set fpga_nconfig output
	--keep it high unless JSM is in UDR and jtag_info_inst is in IR
	-- now we don't need to send DEASSERT od nInitCONF..since it
	-- is low only during UDR state
    enable_nconfig <= jtag_state_e1dr and jtag_info_inst and param2_data_out(1);
    
	ir_out <= ir_in;
end architecture rtl;

