library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;

entity custom_cfg_counter is
	generic
		(
			WIDTH       : natural := 22
		);
	port
		(
            clk         : in STD_LOGIC;
            p_load      : in STD_LOGIC;
            s_in        : in STD_LOGIC;
            p_in        : in STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0);
            count       : in STD_LOGIC;

            p_out       : out STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0)
        );
end entity custom_cfg_counter;

architecture rtl of custom_cfg_counter is
    signal data_reg     : STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0);
begin
    process(clk, s_in, count, p_load, p_in)
    begin
		if ( rising_edge(clk) ) then
           	if (p_load = '1') then
				data_reg(WIDTH-2 downto 0) <= p_in(WIDTH-2 downto 0);
                data_reg(WIDTH-1) <= '0';
   	        elsif (count='1') then
       	        data_reg <= data_reg + conv_std_logic_vector(1, WIDTH);
           	end if;
        end if;
    end process;
    p_out <= data_reg;
end architecture rtl;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;
entity alt_pfl_cfg is
  
  generic
  (
    ADDR_WIDTH				: natural;	-- from 19 for 8Mbits, up to 25 for 512Mbits
    OPTION_START_ADDR		: natural;
    CLK_DIVISOR				: natural := 1;
    CONF_DATA_WIDTH			: natural := 1;		-- valid values are 1 (PS) and 8 (FPP)
    FLASH_DATA_WIDTH		: natural := 16;   -- valid values are 8 and 16
	SAFE_MODE_HALT			: natural := 0;
	SAFE_MODE_RETRY			: natural := 1;
	SAFE_MODE_REVERT		: natural := 0;
	SAFE_MODE_REVERT_ADDR	: natural := 0
  );
  port
  (
    pfl_clk			: in STD_LOGIC;
    pfl_nreset		: in STD_LOGIC;
    pfl_flash_access_granted: in STD_LOGIC;
    pfl_flash_access_request: out STD_LOGIC;

-- 	flash output
    flash_addr		: out STD_LOGIC_VECTOR (ADDR_WIDTH-1 downto 0);
    flash_data_in	: in STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
	flash_read		: out STD_LOGIC;
	flash_select	: out STD_LOGIC;

    page_sel		: in STD_LOGIC_VECTOR (2 DOWNTO 0);
		
--  FPGA
    fpga_conf_done	: in STD_LOGIC;
    fpga_nstatus	: in STD_LOGIC;
    dclk			: out STD_LOGIC;
    fpga_data		: out STD_LOGIC_VECTOR (CONF_DATA_WIDTH-1 downto 0);
    fpga_nconfig	: out STD_LOGIC;

--  from PGM
    enable_configuration : in STD_LOGIC;
    enable_nconfig : in STD_LOGIC
  );
end entity alt_pfl_cfg;

architecture rtl of alt_pfl_cfg is
	function log2(A: integer) return integer is
	begin
	  for I in 1 to 30 loop  -- Works for up to 32 bit integers
	    if(2**I > A) then return(I-1);  end if;
	  end loop;
	  return(30);
	end;
	function max(A, B: integer) return integer is
	begin
		if A>B then return(A); else return(B); end if;
	end;
	COMPONENT lpm_shiftreg
		GENERIC 
		(
			LPM_WIDTH		: POSITIVE;
			LPM_AVALUE		: STRING := "UNUSED";
--			LPM_SVALUE		: NATURAL := 0;
			LPM_PVALUE		: STRING := "UNUSED";
			LPM_DIRECTION	: STRING := "UNUSED";
			LPM_TYPE		: STRING := "LPM_SHIFTREG";
			LPM_HINT		: STRING := "UNUSED"
		);
		PORT 
		(
			data			: IN STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0) := (OTHERS => '0');
			clock			: IN STD_LOGIC;
			enable, shiftin	: IN STD_LOGIC := '1';
			load, sclr, sset, aclr, aset: IN STD_LOGIC := '0';
			q				: OUT STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0);
			shiftout		: OUT STD_LOGIC
		);
	END COMPONENT lpm_shiftreg;
	COMPONENT lpm_counter
		GENERIC (
			LPM_WIDTH           : POSITIVE;
			LPM_MODULUS         : NATURAL   := 0;
			LPM_DIRECTION       : STRING    := "UNUSED";
			LPM_AVALUE          : STRING    := "UNUSED";
			LPM_SVALUE          : STRING    := "UNUSED";
			LPM_PVALUE          : STRING    := "UNUSED";
			LPM_TYPE            : STRING    := "LPM_COUNTER";
			LPM_HINT            : STRING    := "UNUSED");
		PORT (
			data		: IN STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0) := (OTHERS => '0');
			clock		: IN STD_LOGIC;
			clk_en, cnt_en, updown: IN STD_LOGIC := '1';
			sload, sset, sclr, aload, aset, aclr, cin: IN STD_LOGIC := '0';
			cout		: OUT STD_LOGIC;
			q			: OUT STD_LOGIC_VECTOR(LPM_WIDTH-1 DOWNTO 0));
	END COMPONENT;
  	COMPONENT custom_cfg_counter
    	generic
        	(
            	WIDTH       : natural
    	    );
    	port
        	(
            	clk     	: in STD_LOGIC;
    	        p_load      : in STD_LOGIC;
        	    s_in        : in STD_LOGIC;
            	p_in        : in STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0);
    	        count       : in STD_LOGIC;
        	    p_out       : out STD_LOGIC_VECTOR (WIDTH-1 DOWNTO 0)
        	);
	END COMPONENT custom_cfg_counter;	
	
	type CFG_STATE is (CFG_INIT, CFG_VERSION_ADDR, CFG_VERSION_READ, CFG_OPTION_ADDR, CFG_OPTION_READ, CFG_START, CFG_STARTWAIT, CFG_READ, CFG_SHIFT, CFG_WAIT, CFG_USERMODE, CFG_ERROR, CFG_ERRORWAIT);
	signal current_state : CFG_STATE;
	-- Configuration state machine explanation:
	--	CFG_INIT:  The poweron state.  Sets the configuration read address to the option bits
	--		Continues to CFG_OPTION_ADDR when reset deasserted and flash access is granted
	--  CFG_VERSION_ADDR:
	--  CFG_VERSION_READ:
	--	CFG_OPTION_ADDR:  Wait state used to insure the setup time for the flash option bits read
	--		Continues to CFG_OPTION_READ when clock divisor finishes
	--	CFG_OPTION_READ:  Latches in data from the flash into the option bits, increments address.  
	--		Always one clock cycle
	--		Loops back to CFG_OPTION_ADDR until done with option read, then goes to CFG_START
	--	CFG_START:  Wait state used to insure the setup time for the first flash data read
	--		Continues to CFG_READ when clock divisor finishes
	--	CFG_READ:  Latches in a byte or word of flash data
	--		Always one clock cycle, continues to CFG_SHIFT
	--	CFG_SHIFT:  Shifts data out to fpga, either in words or bits, also insures 
	--		the setup time for the next flash read
	--		Waits for shift finished and clock divisor finished
	--		Loops back to CFG_READ until flash address overflow or end of page
	--		On end of page goes to CFG_WAIT, overflow goes to CFG_ERROR
	--	CFG_WAIT:  Wait state for fpga_conf_done
	--		Loops until fpga_conf_done, goes to CFG_USERMODE if asserted or CFG_ERROR if timeout
	--	CFG_USERMODE:  End state
	--	CFG_ERROR:  Resets flash read address to safe address
	--		Continues to CFG_ERRORWAIT
	--	CFG_ERRORWAIT:  Holds nconfig low
	--		Loops for 10uS, continues to CFG_READ

	-- cfg flash data reg
	signal cfg_flash_data_out	: STD_LOGIC_VECTOR(FLASH_DATA_WIDTH-1 DOWNTO 0);
	signal cfg_flash_data_shift	: STD_LOGIC;

	constant OPTION_VERSION_OFFSET : natural := 128; -- 1024 bits
	constant option_register_length : natural := 32;
	constant option_register_reads : natural := option_register_length/FLASH_DATA_WIDTH;
	signal option_bits				: STD_LOGIC_VECTOR (option_register_length-1 DOWNTO 0);
	signal option_read_done			: STD_LOGIC;
	signal option_page_done_bit		: STD_LOGIC;
	signal option_counter_out		: STD_LOGIC_VECTOR(1 DOWNTO 0);
			
	signal flash_addr_overflow		: STD_LOGIC;
	signal flash_addr_at_page_end	: STD_LOGIC;

	--cfg addr counter
	signal cfg_pgm_addr_parallel_in	: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_option_version_addr	: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_page_option_addr		: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_page_start_addr		: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_page_end_addr		: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_page_end_byte_addr	: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_flash_addr_bus		: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_safe_revert_addr		: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);
	signal cfg_byte_addr_parallel_in: STD_LOGIC_VECTOR(ADDR_WIDTH DOWNTO 0);

	-- CFG
	constant cfg_shift_count_width: natural := log2(FLASH_DATA_WIDTH)-log2(CONF_DATA_WIDTH);		-- 4 for PS/16 bits, 0 for FPP/16 bits, 3 for PS/8 bits, 0 for FPP/8 bits
	constant cfg_shift_count_width_signal: natural := max(cfg_shift_count_width, 1);		-- at least 1, to avoid synthesis errors on empty bus...
	signal cfg_shift_count_out		: STD_LOGIC_VECTOR (cfg_shift_count_width_signal DOWNTO 0);
	signal cfg_shift_done			: STD_LOGIC;
	signal cfg_last_shift			: STD_LOGIC;
	signal cfg_shifting				: STD_LOGIC;
	signal cfg_enable_shift			: STD_LOGIC;

	signal enable_dclk				: STD_LOGIC;
	
	signal enable_fpga_config		: STD_LOGIC;
	signal dclk_out					: STD_LOGIC;

	signal cfg_init_signal			: STD_LOGIC;
	signal cfg_version_addr_signal	: STD_LOGIC;
	signal cfg_version_read_signal	: STD_LOGIC;
	signal cfg_option_addr_signal	: STD_LOGIC;
	signal cfg_option_read_signal	: STD_LOGIC;
	signal cfg_start_signal			: STD_LOGIC;
	signal cfg_startwait_signal		: STD_LOGIC;
	signal cfg_shift_signal			: STD_LOGIC;
	signal cfg_read_signal			: STD_LOGIC;
	signal cfg_wait_signal			: STD_LOGIC;
	signal cfg_usermode_signal		: STD_LOGIC;
	signal cfg_error_signal			: STD_LOGIC;
	signal cfg_errorwait_signal			: STD_LOGIC;

	constant CLK_DIVIDER_WIDTH		: NATURAL := log2(CLK_DIVISOR-1)+1;
	signal clk_divider_out			: STD_LOGIC_VECTOR (CLK_DIVIDER_WIDTH-1 downto 0);
	signal clk_divided				: STD_LOGIC;

	signal reset_clock_divider 		: STD_LOGIC;	

	signal fpga_data_signal			: STD_LOGIC_VECTOR(CONF_DATA_WIDTH-1 downto 0);
	
	constant timer_width			: NATURAL := 11;
	signal reset_timer				: STD_LOGIC;
	signal timer_overflow			: STD_LOGIC;
	signal timer_out				: STD_LOGIC_VECTOR(timer_width downto 0);

	signal safe_mode				: STD_LOGIC;
	signal fpga_nstatus_sync		: STD_LOGIC;
	signal fpga_conf_done_sync 		: STD_LOGIC;
	
	signal flash_image_version2		: STD_LOGIC;
	signal flash_image_version2_sig	: STD_LOGIC;
begin
	option_page_done_bit <= option_bits(0);
	
  	-- sync up asynchrouns signals
	process(pfl_clk)
	begin
		if rising_edge(pfl_clk) then
			--enable_fpga_config <= enable_configuration and pfl_flash_access_granted and not fpga_conf_done and fpga_nstatus;
			enable_fpga_config <= enable_configuration;
			fpga_conf_done_sync <= fpga_conf_done;
			fpga_nstatus_sync <= fpga_nstatus;
		end if;
	end process;

	--  Is this really right?  We are dividing by 2^CLK_DIVISOR, not CLK_DIVISOR
	CLK_DIV: if (CLK_DIVISOR>1) generate
		clk_divider: lpm_counter
			generic map 
			(
				LPM_WIDTH => CLK_DIVIDER_WIDTH
			)
			port map
			(
				clock	=> pfl_clk,
				cnt_en	=> not clk_divided,
				sclr	=> reset_clock_divider,
				q		=> clk_divider_out
			);
		clk_divided <= '1' when clk_divider_out = CONV_STD_LOGIC_VECTOR(CLK_DIVISOR-1, CLK_DIVIDER_WIDTH) else '0';
	end generate;
	CLK_DIV_1: if (CLK_DIVISOR = 1) generate
		clk_divided <= '1';
	end generate;
	reset_clock_divider <= cfg_init_signal or cfg_version_read_signal or cfg_option_read_signal or cfg_read_signal;

	clk_timer: lpm_counter
		generic map
		(
			LPM_WIDTH => timer_width+1
		)
		port map
		(
			clock 	=> pfl_clk,
			cnt_en	=> not timer_overflow,
			sclr 	=> reset_timer,
			q		=> timer_out
		);
	timer_overflow <= timer_out(timer_width);
	reset_timer <= cfg_version_read_signal or cfg_option_read_signal or cfg_read_signal or cfg_error_signal or cfg_start_signal;

	process(pfl_clk, cfg_version_read_signal, flash_data_in)
	begin
		if (flash_data_in(7 downto 0) = (8 => '1')) then
			flash_image_version2_sig <= '0';
		else
			flash_image_version2_sig <= '1';
		end if;

		if (rising_edge(pfl_clk)) then
			if (cfg_version_read_signal='1') then
				flash_image_version2 <= flash_image_version2_sig;
			end if;
		end if;
	end process;

	cfg_safe_revert_addr(12 DOWNTO 0) <= (others => '0');
	cfg_safe_revert_addr(ADDR_WIDTH DOWNTO 13) <= CONV_STD_LOGIC_VECTOR(SAFE_MODE_REVERT_ADDR/8192, ADDR_WIDTH-12);
		
	cfg_option_version_addr(12 downto 0) <= CONV_STD_LOGIC_VECTOR(OPTION_VERSION_OFFSET,13);
	cfg_option_version_addr(ADDR_WIDTH downto 13) <= CONV_STD_LOGIC_VECTOR(OPTION_START_ADDR/8192, ADDR_WIDTH-12);

	process (flash_image_version2_sig, page_sel)
	begin
		if (flash_image_version2_sig = '0') then
			cfg_page_option_addr(0) <= '0';
			cfg_page_option_addr(3 DOWNTO 1) <= page_sel;
			cfg_page_option_addr(4) <= '0';
		else
			cfg_page_option_addr(1 downto 0) <= (others => '0');
			cfg_page_option_addr(4 DOWNTO 2) <= page_sel;
		end if;
	end process;
		
	cfg_page_option_addr(12 DOWNTO 5) <= (others => '0');
	cfg_page_option_addr(ADDR_WIDTH DOWNTO 13) <= CONV_STD_LOGIC_VECTOR(OPTION_START_ADDR/8192, ADDR_WIDTH-12);

	cfg_page_start_addr(12 DOWNTO 0) <= (others => '0');
	cfg_page_start_addr(ADDR_WIDTH DOWNTO 13) <= option_bits(ADDR_WIDTH-12 DOWNTO 1);
	cfg_page_end_byte_addr(12 DOWNTO 0) <= (others => '0');
	cfg_page_end_byte_addr(ADDR_WIDTH DOWNTO 13) <= option_bits(16+ADDR_WIDTH-12 DOWNTO 17);

	process (cfg_init_signal, cfg_error_signal, cfg_option_version_addr, cfg_page_option_addr, cfg_page_start_addr, cfg_safe_revert_addr, cfg_version_read_signal)
	begin
		if (cfg_init_signal = '1') then
			cfg_byte_addr_parallel_in <= cfg_option_version_addr;
		elsif (cfg_version_read_signal = '1') then
			cfg_byte_addr_parallel_in <= cfg_page_option_addr;
		elsif (cfg_error_signal = '1') then
			if (SAFE_MODE_REVERT = 1) then
				cfg_byte_addr_parallel_in <= cfg_safe_revert_addr;
			else
				cfg_byte_addr_parallel_in <= cfg_page_start_addr;
			end if;
		else
			cfg_byte_addr_parallel_in <= cfg_page_start_addr;
		end if;
	end process;

	process (cfg_byte_addr_parallel_in, cfg_page_end_byte_addr)
	begin
		if (FLASH_DATA_WIDTH = 8) then
			cfg_pgm_addr_parallel_in <= cfg_byte_addr_parallel_in;
			cfg_page_end_addr <= cfg_page_end_byte_addr;
		else
			cfg_pgm_addr_parallel_in(ADDR_WIDTH) <= '0';
			cfg_pgm_addr_parallel_in(ADDR_WIDTH-1 DOWNTO 0) <= cfg_byte_addr_parallel_in(ADDR_WIDTH DOWNTO 1);
			cfg_page_end_addr(ADDR_WIDTH) <= '0';
			cfg_page_end_addr(ADDR_WIDTH-1 DOWNTO 0) <= cfg_page_end_byte_addr(ADDR_WIDTH DOWNTO 1);
		end if;
	end process;

	cfg_addr: custom_cfg_counter
		generic map
		(
			WIDTH 	=> ADDR_WIDTH + 1
		)
		port map
		(		
			clk		=> pfl_clk,
			p_load	=> cfg_init_signal or cfg_error_signal or cfg_version_read_signal or (cfg_option_read_signal and option_read_done),
			p_in	=> cfg_pgm_addr_parallel_in,
			s_in	=> '0',
			count	=> cfg_option_read_signal or cfg_read_signal,

			p_out	=> cfg_flash_addr_bus
		);
	flash_addr_overflow <= '1' when cfg_flash_addr_bus(ADDR_WIDTH-1 DOWNTO 0) = (ADDR_WIDTH => '1') else '0';
	flash_addr_at_page_end <= '1' when flash_image_version2='1' and cfg_flash_addr_bus(ADDR_WIDTH-1 DOWNTO 0) = cfg_page_end_addr else '0';
	
	
	option_read_counter: lpm_counter
		generic map 
		(
			LPM_WIDTH => 2
		)
		port map
		(
			clock	=> pfl_clk,
			cnt_en	=> cfg_option_read_signal,
			sclr	=> cfg_version_read_signal,
			q		=> option_counter_out
		);

	ORFDW8: if (FLASH_DATA_WIDTH=8) generate
		option_read_done <= '1' when option_counter_out(1 downto 0) = "11" else '0';
	end generate;
	
	ORFDW16: if (FLASH_DATA_WIDTH=16) generate
		option_read_done <= '1' when option_counter_out(0) = '1' else '0';
	end generate;


	process (pfl_clk)
	begin
		if (rising_edge(pfl_clk)) then
			if (cfg_option_read_signal='1') then
				OPTIONREADLOOP: for N in 0 to option_register_reads-1 loop
					if option_counter_out(log2(option_register_reads)-1 downto 0) = conv_std_logic_vector(N, log2(option_register_reads)) then
						option_bits((N+1)*FLASH_DATA_WIDTH-1 downto N*FLASH_DATA_WIDTH) <= flash_data_in(FLASH_DATA_WIDTH-1 downto 0);
					end if;
				end loop;
			end if;
		end if;
	end process;
	

 	cfg_flash_datareg: lpm_shiftreg
		generic map
		(
			LPM_WIDTH 		=> FLASH_DATA_WIDTH,
			LPM_DIRECTION	=> "RIGHT"
		)
		port map
		(
			data			=> flash_data_in,
			clock			=> pfl_clk,
			enable			=> cfg_flash_data_shift or cfg_read_signal,
			load			=> cfg_read_signal,
			q				=> cfg_flash_data_out
		);
	flash_read <= cfg_read_signal or cfg_version_read_signal or cfg_option_read_signal;
	flash_select <= cfg_startwait_signal or 
					cfg_read_signal or cfg_shift_signal or 
					cfg_version_addr_signal or cfg_version_read_signal or 
					cfg_option_addr_signal or cfg_option_read_signal;

	CDW1: if (CONF_DATA_WIDTH=1) generate	-- PS
		cfg_flash_data_shift <= cfg_shift_signal and dclk_out;
		fpga_data_signal <= cfg_flash_data_out(0 downto 0);
	end generate;
	CDW8: if (CONF_DATA_WIDTH=8) generate	-- FPP
		cfg_flash_data_shift <= '0';
		fpga_data_signal <= cfg_flash_data_out(FLASH_DATA_WIDTH-1 downto FLASH_DATA_WIDTH-8) when (cfg_shift_count_out(0)='1') else cfg_flash_data_out(7 downto 0);
	end generate;

	--cfg_shift_done <= '1' when cfg_shift_count_out = conv_std_logic_vector(2**cfg_shift_count_width_signal-2, cfg_shift_count_width_signal) else '0';

	-- set the dclk and fpga_data	
	cfg_enable_shift <= cfg_shift_signal or cfg_read_signal;
	fpga_data <= fpga_data_signal when cfg_enable_shift='1' else (others => '1');

	enable_dclk <= cfg_shifting or cfg_wait_signal or cfg_usermode_signal;
	process(pfl_clk, dclk_out, enable_dclk)
	begin
		if rising_edge(pfl_clk) then
			dclk_out <= not dclk_out and enable_dclk;
		end if;	
	end process;

	cfg_shift_count: lpm_counter
		generic map 
		(
			LPM_WIDTH => cfg_shift_count_width_signal+1
		)
		port map
		(
			clock	=> pfl_clk,
			clk_en	=> '1',
			cnt_en	=> cfg_shifting and dclk_out,
			sclr	=> not cfg_shift_signal,
			q		=> cfg_shift_count_out
		);
	cfg_shifting <= (cfg_shift_signal) and not cfg_shift_done;
	SHIFTCOUNT1: if (cfg_shift_count_width > 0) generate
		cfg_shift_done <= '1' when cfg_shift_count_out(cfg_shift_count_width_signal) = '1' else '0';
		cfg_last_shift <= '1' when cfg_shift_count_out(cfg_shift_count_width_signal-1 downto 0) = (cfg_shift_count_width_signal => '1') else '0';
	end generate;
	SHIFTCOUNT0: if (cfg_shift_count_width = 0) generate
		cfg_shift_done <= '1' when cfg_shift_count_out(cfg_shift_count_width_signal-1 downto 0) = (cfg_shift_count_width_signal => '1') else '0';
		cfg_last_shift <= '1';
	end generate;
	dclk <= dclk_out;
	fpga_nconfig <= '0' when enable_nconfig = '1' or cfg_errorwait_signal = '1' else 'Z';
	flash_addr <= cfg_flash_addr_bus(ADDR_WIDTH-1 downto 0);
	pfl_flash_access_request <= enable_configuration and not cfg_usermode_signal;

	process(pfl_clk, cfg_init_signal, cfg_error_signal)
	begin
		if rising_edge(pfl_clk) then
			if (cfg_init_signal = '1') then
				safe_mode <= '0';
			elsif (cfg_error_signal = '1') then
				safe_mode <= '1';
			end if;
		end if;
	end process;
   
	CFG_STATE_TRANSITION: process(pfl_clk, pfl_nreset, enable_fpga_config)
	begin
		if(pfl_nreset = '0') then
			current_state <= CFG_INIT;
		else
			if rising_edge(pfl_clk) then
				if(enable_fpga_config = '0') then
					current_state <= CFG_INIT;
				else
				case current_state is
					when CFG_INIT =>
						if (pfl_flash_access_granted='1') then
							current_state <= CFG_VERSION_ADDR;
						end if;
					when CFG_VERSION_ADDR =>
						if (clk_divided = '1') then 
							current_state <= CFG_VERSION_READ;
						end if;
					when CFG_VERSION_READ =>
						current_state <= CFG_OPTION_ADDR;
					when CFG_OPTION_ADDR =>
						if (clk_divided = '1') then 
							current_state <= CFG_OPTION_READ;
						end if;
					when CFG_OPTION_READ =>
						if (option_read_done = '1') then
							if (option_page_done_bit = '0') then
								current_state <= CFG_START;
							else
								current_state <= CFG_ERROR;
							end if;
						else
							current_state <= CFG_OPTION_ADDR;
						end if;
					when CFG_START =>
						if (fpga_conf_done_sync = '1') then
							current_state <= CFG_USERMODE;
						elsif (fpga_nstatus_sync = '1' and pfl_flash_access_granted='1' and clk_divided='1') then
							current_state <= CFG_STARTWAIT;
						end if;
					when CFG_STARTWAIT =>
						if (fpga_nstatus_sync = '0') then
							current_state <= CFG_ERROR;
						elsif (timer_overflow = '1') then
							current_state <= CFG_READ;
						end if;
					when CFG_READ =>
						if (fpga_nstatus_sync = '0') then
							current_state <= CFG_ERROR;
						else
							current_state <= CFG_SHIFT;
						end if;
					when CFG_SHIFT =>
						if (fpga_conf_done_sync = '1') then
							current_state <= CFG_USERMODE;
						elsif (fpga_nstatus_sync = '0') then
							current_state <= CFG_ERROR;
						end if;
						if (clk_divided = '1' and (cfg_last_shift = '1' or cfg_shift_done = '1')) then 
							if (flash_addr_overflow = '1' or (safe_mode = '0' and flash_addr_at_page_end = '1')) then
								current_state <= CFG_WAIT;
							else
								current_state <= CFG_READ;
							end if;
						end if;
					when CFG_WAIT =>
						if (fpga_conf_done_sync = '1') then
							current_state <= CFG_USERMODE;
						elsif (fpga_nstatus_sync = '0') then
							current_state <= CFG_ERROR;
						elsif (timer_overflow = '1') then
							current_state <= CFG_ERROR;
						end if;
					when CFG_USERMODE =>
						if (fpga_conf_done_sync = '0') then
							current_state <= CFG_ERROR;
						end if;
					when CFG_ERROR=>
						if (SAFE_MODE_HALT = 0) then
							current_state <= CFG_ERRORWAIT;
						end if;
					when CFG_ERRORWAIT =>
						if timer_overflow = '1' then
							if (SAFE_MODE_RETRY = 1) then
								current_state <= CFG_INIT;
							else
								current_state <= CFG_START;
							end if;
						end if;
					when others =>
						current_state <= CFG_ERROR;
				end case;
				end if;
			end if;
		end if;
	end process CFG_STATE_TRANSITION;

	-- state signals assignments
	process(current_state)
	begin
		if(current_state = CFG_INIT) then
			cfg_init_signal <= '1';
		else
			cfg_init_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_VERSION_ADDR) then
			cfg_version_addr_signal <= '1';
		else
			cfg_version_addr_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_VERSION_READ) then
			cfg_version_read_signal <= '1';
		else
			cfg_version_read_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_OPTION_ADDR) then
			cfg_option_addr_signal <= '1';
		else
			cfg_option_addr_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_OPTION_READ) then
			cfg_option_read_signal <= '1';
		else
			cfg_option_read_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_START) then
			cfg_start_signal <= '1';
		else
			cfg_start_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_STARTWAIT) then
			cfg_startwait_signal <= '1';
		else
			cfg_startwait_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_SHIFT) then
			cfg_shift_signal <= '1';
		else
			cfg_shift_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_READ) then
			cfg_read_signal <= '1';
		else
			cfg_read_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_WAIT) then
			cfg_wait_signal <= '1';
		else
			cfg_wait_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_USERMODE) then
			cfg_usermode_signal <= '1';
		else
			cfg_usermode_signal <= '0';
		end if;
	end process;

	process(current_state)
	begin
		if(current_state = CFG_ERROR) then
			cfg_error_signal <= '1';
		else
			cfg_error_signal <= '0';
		end if;

	end process;
	process(current_state)
	begin
		if(current_state = CFG_ERRORWAIT) then
			cfg_errorwait_signal <= '1';
		else
			cfg_errorwait_signal <= '0';
		end if;
	end process;
end architecture rtl;
