library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_SIGNED.all;

entity alt_pfl is
	generic 
	(
		PFL_IR_BITS		: natural := 5;
		REG_LENGTH		: natural := 16;
		ADDR_WIDTH		: natural;	-- from 19 for 8Mbits, up to 25 for 512Mbits
		OPTION_START_ADDR: natural;
		CLK_DIVISOR		: natural := 1;
		CONF_DATA_WIDTH	: natural := 1;		-- valid values are 1 (PS) and 8 (FPP)
		FLASH_DATA_WIDTH: natural := 16;		-- valid values are 8 and 16
		FEATURES_PGM	: natural := 1;
		FEATURES_CFG	: natural := 1;
		SAFE_MODE_HALT			: natural := 0;
		SAFE_MODE_RETRY			: natural := 1;
		SAFE_MODE_REVERT		: natural := 0;
		SAFE_MODE_REVERT_ADDR	: natural := 0
	);
	port
	(
		pfl_clk			: in STD_LOGIC;
		pfl_nreset		: in STD_LOGIC;
		pfl_flash_access_granted: in STD_LOGIC;
		pfl_flash_access_request: out STD_LOGIC;

-- 		flash output
		flash_addr		: out STD_LOGIC_VECTOR (ADDR_WIDTH-1 downto 0);
		flash_data		: inout STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
		flash_nce		: out STD_LOGIC;
		flash_noe		: out STD_LOGIC;
		flash_nwe		: out STD_LOGIC;
		
--		FPGA
		fpga_pgm		: in STD_LOGIC_VECTOR (2 DOWNTO 0);
		fpga_conf_done	: in STD_LOGIC;
		fpga_nstatus	: in STD_LOGIC;
		dclk			: out STD_LOGIC;
		fpga_data		: out STD_LOGIC_VECTOR (CONF_DATA_WIDTH-1 downto 0);
		fpga_nconfig	: out STD_LOGIC
	);
end entity alt_pfl;

architecture rtl of alt_pfl is
  COMPONENT alt_pfl_cfg is
  
  generic
  (
    ADDR_WIDTH				: natural;	-- from 19 for 8Mbits, up to 25 for 512Mbits
    OPTION_START_ADDR		: natural;
    CLK_DIVISOR				: natural := 1;
    CONF_DATA_WIDTH			: natural := 1;		-- valid values are 1 (PS) and 8 (FPP)
    FLASH_DATA_WIDTH		: natural := 16;   -- valid values are 8 and 16
	SAFE_MODE_HALT			: natural := 0;
	SAFE_MODE_RETRY			: natural := 1;
	SAFE_MODE_REVERT		: natural := 0;
	SAFE_MODE_REVERT_ADDR	: natural := 0
  );
  port
  (
    pfl_clk			: in STD_LOGIC;
    pfl_nreset		: in STD_LOGIC;
    pfl_flash_access_granted: in STD_LOGIC;
    pfl_flash_access_request: out STD_LOGIC;

-- 	flash output
    flash_addr		: out STD_LOGIC_VECTOR (ADDR_WIDTH-1 downto 0);
    flash_data_in	: in STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
    flash_read		: out STD_LOGIC;
	flash_select	: out STD_LOGIC;
			
	page_sel		: in STD_LOGIC_VECTOR (2 DOWNTO 0);

--  FPGA
    fpga_conf_done	: in STD_LOGIC;
    fpga_nstatus	: in STD_LOGIC;
    dclk			: out STD_LOGIC;
    fpga_data		: out STD_LOGIC_VECTOR (CONF_DATA_WIDTH-1 downto 0);
    fpga_nconfig	: out STD_LOGIC;

--  from PGM
    enable_configuration : in STD_LOGIC;
    enable_nconfig : in STD_LOGIC
  );
end COMPONENT alt_pfl_cfg;
COMPONENT alt_pfl_pgm is
	generic 
	(
		PFL_IR_BITS		: natural;
		ADDR_WIDTH		: natural;	-- from 19 for 8Mbits, up to 25 for 512Mbits
		FLASH_DATA_WIDTH: natural	-- valid values are 8 and 16
	);
	port
	(
		pfl_clk			: in STD_LOGIC;
		pfl_nreset		: in STD_LOGIC;
		pfl_flash_access_granted: in STD_LOGIC;
		pfl_flash_access_request: out STD_LOGIC;

--      to CFG
        enable_configuration : out STD_LOGIC;
        enable_nconfig : out STD_LOGIC;
        
-- 		flash output
		flash_addr		: out STD_LOGIC_VECTOR (ADDR_WIDTH-1 downto 0);
		flash_data_in	: in STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
		flash_data_out	: out STD_LOGIC_VECTOR (FLASH_DATA_WIDTH-1 downto 0);
		flash_read		: out STD_LOGIC;
		flash_write		: out STD_LOGIC;
		flash_select	: out STD_LOGIC
	    );
end COMPONENT alt_pfl_pgm;

	signal enable_configuration : STD_LOGIC;
	signal enable_nconfig : STD_LOGIC;
	signal pgm_flash_access_granted : STD_LOGIC;
	signal pgm_flash_access_request : STD_LOGIC;
	signal pgm_flash_data_in : STD_LOGIC_VECTOR(FLASH_DATA_WIDTH-1 downto 0);
	signal pgm_flash_data_out : STD_LOGIC_VECTOR(FLASH_DATA_WIDTH-1 downto 0);
	signal pgm_flash_addr : STD_LOGIC_VECTOR(ADDR_WIDTH-1 downto 0);
	signal pgm_flash_read : STD_LOGIC;
	signal pgm_flash_write : STD_LOGIC;
	signal pgm_flash_select: STD_LOGIC;
	signal cfg_flash_access_granted : STD_LOGIC;
	signal cfg_flash_access_request : STD_LOGIC;
	signal cfg_flash_data_in : STD_LOGIC_VECTOR(FLASH_DATA_WIDTH-1 downto 0);
	signal cfg_flash_addr : STD_LOGIC_VECTOR(ADDR_WIDTH-1 downto 0);
	signal cfg_flash_read : STD_LOGIC;
	signal cfg_flash_select: STD_LOGIC;

	signal flash_data_out : STD_LOGIC_VECTOR(FLASH_DATA_WIDTH-1 downto 0);
	signal flash_read : STD_LOGIC;
	signal flash_write : STD_LOGIC;
	signal flash_select: STD_LOGIC;

begin
PGM1: if (FEATURES_PGM = 1) generate
  pgm: alt_pfl_pgm
    generic map
    (
      PFL_IR_BITS => PFL_IR_BITS,
      ADDR_WIDTH => ADDR_WIDTH,
      FLASH_DATA_WIDTH => FLASH_DATA_WIDTH
    )
    port map
    (
      pfl_clk => pfl_clk,
      pfl_nreset => pfl_nreset,
      pfl_flash_access_granted => pgm_flash_access_granted,
      pfl_flash_access_request => pgm_flash_access_request,
      flash_addr => pgm_flash_addr,
      flash_data_in => flash_data,
      flash_data_out => pgm_flash_data_out,
      flash_read => pgm_flash_read,
      flash_write => pgm_flash_write,
      flash_select => pgm_flash_select,
      enable_configuration => enable_configuration,
      enable_nconfig => enable_nconfig
      );
end generate;
PGM0: if (FEATURES_PGM = 0) generate
	  pgm_flash_access_request <= '0';
	  enable_configuration <= '1';
	  enable_nconfig <= '0';
end generate;
CFG1: if (FEATURES_CFG = 1) generate
	cfg: alt_pfl_cfg
    generic map
    (
		ADDR_WIDTH => ADDR_WIDTH,
		OPTION_START_ADDR => OPTION_START_ADDR,
		CLK_DIVISOR => CLK_DIVISOR,
		CONF_DATA_WIDTH => CONF_DATA_WIDTH,
		FLASH_DATA_WIDTH => FLASH_DATA_WIDTH,
		SAFE_MODE_HALT => SAFE_MODE_HALT,
		SAFE_MODE_RETRY => SAFE_MODE_RETRY,
		SAFE_MODE_REVERT => SAFE_MODE_REVERT,
		SAFE_MODE_REVERT_ADDR => SAFE_MODE_REVERT_ADDR
    )
    port map
    (
      pfl_clk => pfl_clk,
      pfl_nreset => pfl_nreset,
      pfl_flash_access_granted => cfg_flash_access_granted,
      pfl_flash_access_request => cfg_flash_access_request,
      flash_addr => cfg_flash_addr,
      flash_data_in => flash_data,
      flash_read => cfg_flash_read,
      flash_select => cfg_flash_select,
      page_sel => fpga_pgm,
      fpga_conf_done => fpga_conf_done,
      fpga_nstatus => fpga_nstatus,
      dclk => dclk,
      fpga_data => fpga_data,
      fpga_nconfig => fpga_nconfig,
      enable_configuration => enable_configuration,
      enable_nconfig => enable_nconfig
      );
  end generate;
  CFG0: if (FEATURES_CFG = 0) generate
	  cfg_flash_access_request <= '0';
  end generate;

  process (pgm_flash_access_request, cfg_flash_access_request, pgm_flash_access_granted, cfg_flash_access_granted, pfl_flash_access_granted, pgm_flash_addr, pgm_flash_data_out, pgm_flash_read, pgm_flash_write, cfg_flash_addr, cfg_flash_read)
    begin
      if (pgm_flash_access_request = '1') then
        pfl_flash_access_request <= pgm_flash_access_request;
        pgm_flash_access_granted <= pfl_flash_access_granted;
        flash_addr <= pgm_flash_addr;
		flash_data_out <= pgm_flash_data_out;
        flash_read <= pgm_flash_read;
        flash_write <= pgm_flash_write;
        flash_select <= pgm_flash_select;
		cfg_flash_access_granted <= '0';
      elsif (cfg_flash_access_request = '1') then
        pfl_flash_access_request <= cfg_flash_access_request;
        cfg_flash_access_granted <= pfl_flash_access_granted;
        flash_addr <= cfg_flash_addr;
        flash_read <= cfg_flash_read;
		flash_write <= '0';
        flash_select <= cfg_flash_select;
		flash_data_out <= (others => 'Z');
		pgm_flash_access_granted <= '0';
      else
        pfl_flash_access_request <= '0';
		flash_read <= '0';
		flash_write <= '0';
        flash_select <= '0';
		flash_addr <= (others => 'X');
		flash_data_out <= (others => 'X');
		pgm_flash_access_granted <= '0';
		cfg_flash_access_granted <= '0';
      end if;
    end process;
	-- generates control signal
	flash_nce <= not flash_select;
	ctrl_sig : process(flash_read, flash_write, flash_data_out, flash_select)
	begin
		if (flash_read = '1') then           -- read operation
			flash_nwe <= '1';
			flash_noe <= '0';
			flash_data <= (others => 'Z');
		elsif (flash_write = '1') then		-- write operation
			flash_nwe <= '0';
			flash_noe <= '1';
			flash_data <= flash_data_out;
		else											-- no flash access
			flash_nwe <= '1';
			flash_noe <= '1';
			flash_data <= (others => 'X');
			--flash_data <= (others => 'Z');
		end if;
	end process ctrl_sig;
end architecture rtl;
