/*
 * This module encapsulates the Iota ABB chip plus the following:
 *
 * - bypass capacitors on the VBAT input and the regulator outputs
 * - all GND connections
 * - IBIAS and VREF external components
 * - UPR, VLMEM, VLRTC and everything connected to them
 * - pull-up of SIM_IO to VSIM
 * - cap on the AFC output
 * - RC network joining BDL[IQ][MP] and BUL[IQ][MP]
 *
 * All other Iota signals are passed through untouched.
 */

module abb_block (GND, VBAT, VSIM, Vdbb, Vio, Vflash, Vsram, Vrtc,
		  nRESPWON, nTESTRESET,
		  Analog_IM, Analog_IP, Analog_QM, Analog_QP,
		  ADIN1, ADIN2, ADIN3, ADIN4, AFC, APC, AUXI, AUXON, AUXOP,
		  BDR, BDX, BFSR, BFSX, CK13M, CK32K, DAC, DBBSCK, DBBSIO,
		  DBBSRST, EARN, EARP,
		  HSMICBIAS, HSMICP, HSO, ICTL, INT1, INT2, ITWAKEUP,
		  LED_A, LED_B, LED_C, MICBIAS, MICIN, MICIP, ON_nOFF,
		  PCHG, PWON, RPWON,
		  TCK, TDI, TDO, TDR, TEN, TMS, UDR, UDX, UEN,
		  VBACKUP, VBATS, VCCS, VCHG, VCK, VDR, VDX, VFS,
		  SIM_IO, SIM_CLK, SIM_RST);

input GND, VBAT;
output VSIM, Vdbb, Vio, Vflash, Vsram, Vrtc;

output nRESPWON;
input nTESTRESET;

inout Analog_IM, Analog_IP, Analog_QM, Analog_QP;

input ADIN1, ADIN2, ADIN3, ADIN4;
output AFC, APC, DAC;

input AUXI;
output AUXON, AUXOP;
output EARN, EARP;
output HSMICBIAS, HSO;
input HSMICP;
output MICBIAS;
input MICIN, MICIP;

input BDR, BFSR;
output BDX, BFSX;
input TDR, TEN;
input UDR, UEN;
output UDX;
output VCK, VDX, VFS;
input VDR;

input CK13M, CK32K, ITWAKEUP;
output INT1, INT2;
output ON_nOFF;

input DBBSCK, DBBSRST;
inout DBBSIO;

input PWON, RPWON;
output ICTL, PCHG;

output LED_A, LED_B, LED_C;

output SIM_CLK, SIM_RST;
inout SIM_IO;

input TCK, TDI, TMS;
output TDO;

input VBACKUP, VBATS, VCCS, VCHG;

/* nets inside this module */
wire UPR, VLMEM, Vabb;
wire IBIAS, VREF;
wire BULIM, BULIP, BULQM, BULQP;

/* instantiate the Iota! */

iota_100ggm iota (.ADIN1(ADIN1),
		  .ADIN2(ADIN2),
		  .ADIN3(ADIN3),
		  .ADIN4(ADIN4),
		  .AFC(AFC),
		  .APC(APC),
		  .AUXI(AUXI),
		  .AUXON(AUXON),
		  .AUXOP(AUXOP),
		  .BDLIM(Analog_IM),
		  .BDLIP(Analog_IP),
		  .BDLQM(Analog_QM),
		  .BDLQP(Analog_QP),
		  .BDR(BDR),
		  .BDX(BDX),
		  .BFSR(BFSR),
		  .BFSX(BFSX),
		  .BULIM(BULIM),
		  .BULIP(BULIP),
		  .BULQM(BULQM),
		  .BULQP(BULQP),
		  .CK13M(CK13M),
		  .CK32K(CK32K),
		  .DAC(DAC),
		  .DBBSCK(DBBSCK),
		  .DBBSIO(DBBSIO),
		  .DBBSRST(DBBSRST),
		  .EARN(EARN),
		  .EARP(EARP),
		  .GNDA(GND),
		  .GNDAV(GND),
		  .GNDD(GND),
		  .GNDL1(GND),
		  .GNDL2(GND),
		  .HSMICBIAS(HSMICBIAS),
		  .HSMICP(HSMICP),
		  .HSO(HSO),
		  .IBIAS(IBIAS),
		  .ICTL(ICTL),
		  .INT1(INT1),
		  .INT2(INT2),
		  .ITWAKEUP(ITWAKEUP),
		  .LEDA(LED_A),
		  .LEDB1(LED_B),
		  .LEDB2(LED_B),
		  .LEDC(LED_C),
		  .MICBIAS(MICBIAS),
		  .MICIN(MICIN),
		  .MICIP(MICIP),
		  .ON_nOFF(ON_nOFF),
		  .PCHG(PCHG),
		  .PWON(PWON),
		  .REFGND(GND),
		  .RESPWONz(nRESPWON),
		  .RPWON(RPWON),
		  .SIMCK(SIM_CLK),
		  .SIMIO(SIM_IO),
		  .SIMRST(SIM_RST),
		  .TCK(TCK),
		  .TDI(TDI),
		  .TDO(TDO),
		  .TDR(TDR),
		  .TEN(TEN),
		  .TEST3(),	/* no connect */
		  .TEST4(),	/* ditto */
		  .TESTRSTz(nTESTRESET),
		  .TESTV(),	/* no connect */
		  .TMS(TMS),
		  .UDR(UDR),
		  .UDX(UDX),
		  .UEN(UEN),
		  .UPR(UPR),
		  .VBACKUP(VBACKUP),
		  .VBAT(VBAT),
		  .VBATS(VBATS),
		  .VCABB(VBAT),
		  .VCCS(VCCS),
		  .VCDBB(VBAT),
		  .VCHG(VCHG),
		  .VCIO1(VBAT),
		  .VCIO2(VBAT),
		  .VCK(VCK),
		  .VCMEM(VBAT),
		  .VCRAM(VBAT),
		  .VDR(VDR),
		  .VDX(VDX),
		  .VFS(VFS),
		  .VLMEM(VLMEM),
		  .VLRTC(GND),
		  .VRABB(Vabb),
		  .VRDBB(Vdbb),
		  .VREF(VREF),
		  .VRIO1(Vio),
		  .VRIO2(Vio),
		  .VRMEM(Vflash),
		  .VRRAM(Vsram),
		  .VRRTC(Vrtc),
		  .VRSIM(VSIM),
		  .VSDBB(Vdbb),
		  .VXRTC()	/* no connect */
	);

/* power bypass caps per Leonardo schematics */

/* VBAT input */
capacitor C220 (VBAT, GND);
capacitor C221 (VBAT, GND);

/* regulator outputs */
capacitor C213 (Vabb, GND);
capacitor C214 (Vdbb, GND);
capacitor C215 (Vio, GND);
capacitor C216 (Vflash, GND);
capacitor C217 (Vsram, GND);
capacitor C218 (VSIM, GND);
capacitor C219 (Vrtc, GND);

/* UPR bypass cap */
capacitor C208 (UPR, GND);

/*
 * VLMEM needs to be pulled up to UPR.
 * Leonardo also had a pull-down resistor footprint; I omitted it
 * originally for FCM01, but we have to reinstate it for FCDEV3B
 * as Openmoko's PCB layout has it.
 */
resistor R209 (VLMEM, UPR);
resistor R210 (VLMEM, GND);

/* nTESTRESET also needs to be pulled up to UPR */
resistor R208 (nTESTRESET, UPR);

/* IBIAS and VREF */
resistor R204 (IBIAS, GND);
capacitor C204 (VREF, GND);

/* pull-up on SIM_IO to VSIM */
resistor R206 (SIM_IO, VSIM);

/* cap on AFC output */
capacitor C205 (AFC, GND);

/* RC network joining BDL[IQ][MP] and BUL[IQ][MP] */

abb_rc_network abb_rc_network ( .IM_bidir(Analog_IM),
				.IP_bidir(Analog_IP),
				.QM_bidir(Analog_QM),
				.QP_bidir(Analog_QP),
				.IM_abbout(BULIM),
				.IP_abbout(BULIP),
				.QM_abbout(BULQM),
				.QP_abbout(BULQP)
		);

endmodule
